/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#ifdef HAVE_SQL


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>

#if (WIN32|WINNT)
#include <time.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_spell.h"
#include "udm_robots.h"
#include "udm_mutex.h"
#include "udm_db.h"
#include "udm_unicode.h"
#include "udm_url.h"
#include "udm_log.h"
#include "udm_proto.h"
#include "udm_conf.h"
#include "udm_crc32.h"
#include "udm_xmalloc.h"
#include "udm_cache.h"
#include "udm_boolean.h"
#include "udm_searchtool.h"
#include "udm_searchcache.h"
#include "udm_server.h"
#include "udm_stopwords.h"
#include "udm_doc.h"
#include "udm_result.h"
#include "udm_vars.h"
#include "udm_agent.h"
#include "udm_store.h"
#include "udm_hrefs.h"
#include "udm_word.h"
#include "udm_db_int.h"
#include "udm_sqldbms.h"

#if HAVE_ORACLE8
static void param_add(UDM_DB *db, int pos_out_1, int pos_out_2, int pos_out_3, int pos_out_4){
        db->par->out_pos_val[0][db->par->out_rec] = pos_out_1;
        db->par->out_pos_val[1][db->par->out_rec] = pos_out_2;
        db->par->out_pos_val[2][db->par->out_rec] = pos_out_3;
        db->par->out_pos_val[3][db->par->out_rec] = pos_out_4;
        db->par->out_rec++;
}

static void param_init(UDM_DB *db, int pos1, int pos2, int pos3, int pos4){
        db->par->out_pos[0]=pos1;
        db->par->out_pos[1]=pos2;
        db->par->out_pos[2]=pos3;
        db->par->out_pos[3]=pos4;
}

#endif

static void AutoWild(UDM_AGENT * Agent,UDM_VARLIST * vars){
	int i;
	
	for(i=0;i<vars->nvars;i++){
		if(!strcmp(vars->Var[i].name,"ul")){
			if((vars->Var[i].val)&&(vars->Var[i].val[0])){
				UDM_URL url;
				char * wild;
				
				wild=(char*)malloc(strlen(vars->Var[i].val)+3);
				UdmURLParse(&url,vars->Var[i].val);
				if(url.schema[0]){
					sprintf(wild,"%s%%",vars->Var[i].val);
				}else{
					sprintf(wild,"%%%s%%",vars->Var[i].val);
				}
				free(vars->Var[i].val);
				vars->Var[i].val=wild;
			}
		}
	}
}




static const char *BuildWhere(UDM_ENV * Conf,UDM_DB *db){
	size_t	i;
	char	*urlstr;
	char	*tagstr;
	char	*statusstr;
	char	*catstr;
	char	*langstr;
	
	if(db->where)return(db->where);
	
	urlstr=strdup("");
	tagstr=strdup("");
	statusstr=strdup("");
	catstr=strdup("");
	langstr=strdup("");
	
	
	for(i=0;i<Conf->Vars.nvars;i++){
		const char *var=Conf->Vars.Var[i].name?Conf->Vars.Var[i].name:"";
		const char *val=Conf->Vars.Var[i].val?Conf->Vars.Var[i].val:"";
		int intval=atoi(val);
		
		if(!val[0])continue;
		
		if(!strcmp(var,"tag") || !strcmp(var,"t")){
			tagstr=(char*)realloc(tagstr,strlen(tagstr)+strlen(val)+50);
			if(tagstr[0])strcpy(UDM_STREND(tagstr)-1," OR ");
			else	strcat(tagstr,"(");

			if(db->DBType==UDM_DB_PGSQL)
				sprintf(UDM_STREND(tagstr),"(url.tag || '') LIKE '%s')",val);
			else
				sprintf(UDM_STREND(tagstr),"url.tag LIKE '%s')",val);
		}
		
		if(!strcmp(var,"s")){
			statusstr=(char*)realloc(statusstr,strlen(statusstr)+strlen(val)+50);
			
			if(db->DBSQL_IN){
				if(statusstr[0])sprintf(UDM_STREND(statusstr)-1,",%d)",intval);
				else	sprintf(statusstr," url.status IN (%d)",intval);
			}else{
				if(statusstr[0])strcpy(UDM_STREND(statusstr)-1," OR ");
				else	strcat(statusstr,"(");
				sprintf(UDM_STREND(statusstr),"url.status=%d)",intval);
			}
		}

		if(!strcmp(var,"ul")){
			urlstr=(char*)realloc(urlstr,strlen(urlstr)+strlen(val)+50);
#ifdef FIXME_1
			UDM_URL	URL;
			UdmURLParse(&URL,val);
			if(URL.schema[0] && URL.hostinfo[0]){
				snprintf(site_id_str,sizeof(site_id_str)-1,"%s://%s/",urlstruct.schema,urlstruct.hostinfo);
				sitelimit=UdmStrCRC32(site_id_str);
			}else{
				sitelimit=0;
			}
#endif
			if(urlstr[0])strcpy(UDM_STREND(urlstr)-1," OR ");
			else	strcat(urlstr,"(");
			if(db->DBType==UDM_DB_PGSQL)
				sprintf(UDM_STREND(urlstr),"(url.url || '') LIKE '%s')",val);
			else
				sprintf(UDM_STREND(urlstr),"url.url LIKE '%s')",val);
		}
		
		if(!strcmp(var,"lang")){
			langstr=(char*)realloc(langstr,strlen(langstr)+strlen(val)+50);
			if(langstr[0])strcpy(UDM_STREND(langstr)-1," OR ");
			else	strcat(langstr,"(");
			sprintf(UDM_STREND(langstr),"url.lang LIKE '%s')",val);
		}
		
		if(!strcmp(var,"cat")){
			catstr=(char*)realloc(catstr,strlen(catstr)+strlen(val)+50);
			if(catstr[0])strcpy(UDM_STREND(catstr)-1," OR ");
			else	strcat(catstr,"(");
			sprintf(UDM_STREND(catstr),"url.category LIKE '%s%%')",val);
		}
#ifdef FIXME_1
		switch(stl->type){
			case 1: case -1:
				if (stl->type==1)
					lt_gt='>';
				else
					lt_gt='<';
				sprintf(c->timestr, " AND (url.last_mod_time%c%li)", lt_gt, stl->t1);
				break;
			case 2: /* between */
				sprintf(c->timestr, " AND (url.last_mod_time BETWEEN %li AND %li)", stl->t1, stl->t2);
				break;
			default:
				return -1;
		}
#endif
	}
	

	if( !urlstr[0] && !tagstr[0] && !statusstr[0] && !catstr[0] && !langstr[0] ){
		db->where=strdup("");
		goto ret;
	}
	i=strlen(urlstr)+strlen(tagstr)+strlen(statusstr)
				+strlen(catstr)+strlen(langstr);
	db->where=(char*)malloc(i+100);
	db->where[0] = '\0';
	
	if(urlstr[0]){
		strcat(db->where,urlstr);
	}
	if(tagstr[0]){
		if(db->where[0])strcat(db->where," AND ");
		strcat(db->where,tagstr);
	}
	if(statusstr[0]){
		if(db->where[0])strcat(db->where," AND ");
		strcat(db->where,statusstr);
	}
	if(catstr[0]){
		if(db->where[0])strcat(db->where," AND ");
		strcat(db->where,catstr);
	}
	if(langstr[0]){
		if(db->where[0])strcat(db->where," AND ");
		strcat(db->where,langstr);
	}
ret:
	free(urlstr);
	free(tagstr);
	free(statusstr);
	free(catstr);
	free(langstr);
	return db->where;
}


/********************************************************************/
/* --------------------------------------------------------------- */
/* Almost Unified code for all supported SQL backends              */
/* --------------------------------------------------------------- */
/*******************************************************************/


static const int search_cache_size=1000;


/************* Servers ******************************************/

static char * strdupnull(const char * src){
	if(src){
		if(*src){
			return(strdup(src));
		}else{
			return NULL;
		}
	}else{
		return NULL;
	}
}

static int UdmLoadServerTable(UDM_AGENT * Indexer, UDM_SERVERLIST *S, UDM_DB *db){
	size_t		rows,i;
	UDM_SERVER	Server;
	UDM_HREF	Href;
	UDM_SQLRES	SQLRes;
	char		qbuf[1024];
	const char	*name=db->addr.filename[0]?db->addr.filename:"servers";
	int		res;
	
	snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT rec_id,url,alias,period,tag,category,charset,lang,basic_auth,\
proxy,proxy_auth,\
maxhops,gindex,follow,use_robots,use_clones,\
max_net_errors,net_delay_time,read_timeout \
FROM %s WHERE active=1",name);

	if(UDM_OK!=(res=UdmSQLQuery(db,&SQLRes,qbuf)))
		return res;
	
	bzero(&Href,sizeof(Href));
	
	rows=UdmSQLNumRows(&SQLRes);
	for(i=0;i<rows;i++){
		const char	*val;
		
		UdmServerInit(&Server);
		
		Server.url		=strdupnull(UdmSQLValue(&SQLRes,i,1));
		Server.alias		=strdupnull(UdmSQLValue(&SQLRes,i,2));
		Server.Spider.period	=UDM_ATOI(UdmSQLValue(&SQLRes,i,3));
		
		if((val=UdmSQLValue(&SQLRes,i,4)) && val[0])
			UdmVarListAddStr(&Server.Vars,"Tag",val);
		
		if((val=UdmSQLValue(&SQLRes,i,5)) && val[0])
			UdmVarListAddStr(&Server.Vars,"Category",val);
		
		if((val=UdmSQLValue(&SQLRes,i,6)) && val[0])
			UdmVarListAddStr(&Server.Vars,"Default-Charset",val);
		
		if((val=UdmSQLValue(&SQLRes,i,7)) && val[0])
			UdmVarListAddStr(&Server.Vars,"Default-Content-Language",val);
		
		if((val=UdmSQLValue(&SQLRes,i,8)) && val[0]){
			char	*basic_auth;
			basic_auth=(char*)malloc(BASE64_LEN(strlen(val)));
			udm_base64_encode(val,basic_auth,strlen(val));
			UdmVarListAddStr(&Server.ExtraHeaders,"Authorization",basic_auth);
			UDM_FREE(basic_auth);
		}
		
		if((val=UdmSQLValue(&SQLRes,i,9)) && val[0]){
			UdmVarListAddStr(&Server.ExtraHeaders,"Proxy",val);/* FIXME: separate port */
		}
		
		if((val=UdmSQLValue(&SQLRes,i,10)) && val[0]){
			char	*basic_auth;
			basic_auth=(char*)malloc(BASE64_LEN(strlen(val)));
			udm_base64_encode(val,basic_auth,strlen(val));
			UdmVarListAddStr(&Server.ExtraHeaders,"Proxy-Authorization",basic_auth);
			UDM_FREE(basic_auth);
		}
		
		Server.Spider.maxhops		=UDM_ATOI(UdmSQLValue(&SQLRes,i,11));
		Server.Spider.index		=UDM_ATOI(UdmSQLValue(&SQLRes,i,12));
		Server.Spider.follow		=UDM_ATOI(UdmSQLValue(&SQLRes,i,13));
		Server.Spider.use_robots	=UDM_ATOI(UdmSQLValue(&SQLRes,i,14));
		Server.Spider.use_clones	=UDM_ATOI(UdmSQLValue(&SQLRes,i,15));
		Server.Spider.max_net_errors	=UDM_ATOI(UdmSQLValue(&SQLRes,i,16));
		Server.Spider.net_error_delay_time=UDM_ATOI(UdmSQLValue(&SQLRes,i,17));
		Server.Spider.read_timeout	=UDM_ATOI(UdmSQLValue(&SQLRes,i,18));
		Server.match_type		=UDM_SERVER_SUBSTR|UDM_SERVER_MATCH;
		UdmServerAdd(Indexer->Conf,&Server);
		if(Server.url[0]){
			Href.url=Server.url;
			Href.method=UDM_METHOD_GET;
			UdmHrefListAdd(&Indexer->Conf->Hrefs,&Href);
		}
		UdmServerFree(&Server);
	}
	UdmSQLFree(&SQLRes);
	return(UDM_OK);
}

/************************* find url ********************************/

static int UdmFindURL(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db){
	UDM_SQLRES	SQLRes;
	const char	*url=UdmVarListFindStr(&Doc->Sections,"URL","");
	int		id=0;
	int		use_crc32_url_id=!strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars,"UseCRC32URLId","no"),"yes");
	int		rc=UDM_OK;
	
	if(use_crc32_url_id){
		/* Auto generation of rec_id */
		/* using CRC32 algorythm     */
		id=UdmStrCRC32(url);
	}else{
		size_t i = 127;
		const char *o;
		char *qbuf = NULL;
		char *e_url = NULL;
		
		/* Escape URL string */
		if ((e_url = (char*)malloc( i = (7 * strlen(url)) )) == NULL) return UDM_ERROR;
		if ((qbuf = (char*)malloc( i + 100 )) == NULL){
			UDM_FREE(e_url);
			return UDM_ERROR;
		}
		UdmDBEscStr(db->DBType,e_url,url,strlen(url));
		snprintf(qbuf, i + 100, "SELECT rec_id FROM url WHERE url='%s'",e_url);
		if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,qbuf))){
			UDM_FREE(e_url);
			UDM_FREE(qbuf);
			return rc;
		}
		for(i=0;i<UdmSQLNumRows(&SQLRes);i++){
			if((o=UdmSQLValue(&SQLRes,i,0))){
				id=atoi(o);
				break;
			}
		}
		UDM_FREE(e_url);
		UDM_FREE(qbuf);
		UdmSQLFree(&SQLRes);
	}
	UdmVarListReplaceInt(&Doc->Sections,"ID",id);
	return	rc;
}


static int UdmFindMessage(UDM_AGENT *Indexer, UDM_DOCUMENT * Doc, UDM_DB *db){
	size_t 		i;
	char 		qbuf[1024];
	char 		eid[UDM_URLSIZE*4];
	UDM_SQLRES	SQLRes;
	const char	*message_id=UdmVarListFindStr(&Doc->Sections,"Message-ID",NULL);
	int		rc;
	
	if(!message_id)
		return UDM_OK;
	
	/* Escape URL string */
	UdmDBEscStr(db->DBType,eid,message_id,strlen(message_id));
	
	snprintf(qbuf, 1023, "SELECT rec_id FROM url WHERE msg_id='%s'", eid);
	if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,qbuf)))
		return rc;
	
	for(i=0;i<UdmSQLNumRows(&SQLRes);i++){
		const char * o;
		if((o=UdmSQLValue(&SQLRes,i,0))){
			UdmVarListReplaceInt(&Doc->Sections,"ID",atoi(o));
			break;
		}
	}
	UdmSQLFree(&SQLRes);
	return(UDM_OK);
}


/********************** Words ***********************************/

static int UdmDeleteWordFromURL(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db){
	char	qbuf[512];
	int	i=0,last=0,rc=UDM_OK;
	int	url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
	
	switch(db->DBMode){
	
	case UDM_DBMODE_MULTI:
		for(i=MINDICT;i<MAXDICT;i++){
			if(last!=DICTNUM(i)){
				snprintf(qbuf,sizeof(qbuf),"DELETE FROM dict%d WHERE url_id=%d",DICTNUM(i),url_id);
				if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
					return rc;
				last=DICTNUM(i);
			}
		}
		break;
	case UDM_DBMODE_MULTI_CRC:
		for(i=MINDICT;i<MAXDICT;i++){
			if(last!=DICTNUM(i)){
				int done=0;
#ifdef HAVE_ORACLE8
				if(db->DBDriver==UDM_DB_ORACLE8){
					snprintf(qbuf,sizeof(qbuf)-1,"DELETE FROM ndict%d WHERE url_id=:1", DICTNUM(i),url_id);
					param_init(db, 1, 0, 0, 0);
					param_add(db, url_id, 0, 0, 0);
					done=1;
				}
#endif				
				if(!done)
					snprintf(qbuf,sizeof(qbuf)-1,"DELETE FROM ndict%d WHERE url_id=%d",DICTNUM(i),url_id);

				if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
					return rc;
				
				last=DICTNUM(i);
			}
		}
		break;
	case UDM_DBMODE_SINGLE_CRC:
		snprintf(qbuf,sizeof(qbuf)-1,"DELETE FROM ndict WHERE url_id=%d",url_id);
		if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
			return rc;
		break;
	case UDM_DBMODE_BLOB:
		/*i=UdmBlobWriteStamp(Indexer,Doc);*/
		return(i);
	case UDM_DBMODE_CACHE:
		/* Let's lock it */
		i=UdmDeleteURLFromCache(Indexer,url_id,db);
		return(i);
	default:  /* UDM_DBMODE_SINGLE */
		snprintf(qbuf, sizeof(qbuf)-1, "DELETE FROM dict WHERE url_id=%d", url_id);
		if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
			return(UDM_ERROR);
		break;
	}
	return(UDM_OK);
}

static int StoreWordsMulti(UDM_AGENT * Indexer,UDM_DOCUMENT * Doc,UDM_DB *db){
	char	qbuf[512];
	char	tablename[64]="dict";
	char	tbl_nm[64];
	int	n,prev_dictlen=0,rc;
	int	url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
	
	if(db->DBMode==UDM_DBMODE_MULTI){
		strcpy(tablename,"dict");
	}else{
		strcpy(tablename,"ndict");
	}
	
	for(n=0;n<NDICTS;n++){
		if(prev_dictlen==dictlen[n])continue;
		prev_dictlen=dictlen[n];

		sprintf(tbl_nm, "%s%d", tablename, dictlen[n]);
		if(1){
			switch(db->DBType){
				case UDM_DB_PGSQL:
					rc=UdmSQLQuery(db,NULL,"BEGIN WORK");
					break;
				case UDM_DB_ORACLE7:
				case UDM_DB_ORACLE8:
				case UDM_DB_SAPDB:
					rc=UdmSQLQuery(db,NULL,"COMMIT");
					db->commit_fl = 1;
					break;
				default:
					db->commit_fl = 1;
					break;
			}
			if(rc!=UDM_OK)
				return rc;
		}
		
		/* Delete old words */
		snprintf(qbuf,sizeof(qbuf),"DELETE FROM %s WHERE url_id=%d",tbl_nm,url_id);
		if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
			return rc;
		
		/* Insert new word */
		if(db->DBType==UDM_DB_MYSQL){
			int have_words=0;
			char * qb,*qe;
			size_t step=1024,mlen=1024,len,i;

			qb=(char*)malloc(mlen);
			if(db->DBMode==UDM_DBMODE_MULTI){
				sprintf(qb,"INSERT INTO %s (url_id,word,intag) VALUES ",tbl_nm);
			}else{
				sprintf(qb,"INSERT INTO %s (url_id,word_id,intag) VALUES ",tbl_nm);
			}
			qe=qb+strlen(qb);

			for(i=0;i<Doc->Words.nwords;i++){
				if(!Doc->Words.Word[i].coord)continue;
				
				if (DICTNUM(strlen(Doc->Words.Word[i].word)) == dictlen[n]){
					len=qe-qb;
					/* UDM_MAXWORDSIZE+100 should be enough */
					if((len+UDM_MAXWORDSIZE+100)>=mlen){
						mlen+=step;
						qb=(char*)realloc(qb,mlen);
						qe=qb+len;
					}
					if(have_words)strcpy(qe++,",");
					have_words++;
					if(db->DBMode==UDM_DBMODE_MULTI){
						sprintf(qe,"(%d,'%s',%d)",url_id,Doc->Words.Word[i].word,Doc->Words.Word[i].coord);
					}else{
						sprintf(qe,"(%d,%d,%d)",url_id,UdmStrCRC32(Doc->Words.Word[i].word),Doc->Words.Word[i].coord);
					}
					qe=qe+strlen(qe);
					
					/* Insert 64k packets max to stay */
					/* under MySQL's default limit. */
					if((qe-qb)>=64*1024){
						if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qb))){
							free(qb);
							return rc;
						}
						/* Init query again */
						if(db->DBMode==UDM_DBMODE_MULTI){
							sprintf(qb,"INSERT INTO %s (url_id,word,intag) VALUES ",tbl_nm);
						}else{
							sprintf(qb,"INSERT INTO %s (url_id,word_id,intag) VALUES ",tbl_nm);
						}
						qe=qb+strlen(qb);
						have_words = 0;
					}
				}
			}
			if(have_words)rc=UdmSQLQuery(db,NULL,qb);
			free(qb);
			if(rc!=UDM_OK)return rc;
		}else{
			size_t i;
			for(i=0;i<Doc->Words.nwords;i++){
				int len;
				if(!Doc->Words.Word[i].coord)continue;
				
				len=strlen(Doc->Words.Word[i].word);
				if (DICTNUM(len) == dictlen[n]){
					if(db->DBMode==UDM_DBMODE_MULTI){
						snprintf(qbuf,sizeof(qbuf)-1,"INSERT INTO %s (url_id,word,intag) VALUES(%d,'%s',%d)",tbl_nm,url_id,Doc->Words.Word[i].word,Doc->Words.Word[i].coord);
					}else{
						snprintf(qbuf,sizeof(qbuf)-1,"INSERT INTO %s (url_id,word_id,intag) VALUES(%d,%d,%d)",tbl_nm,url_id,UdmStrCRC32(Doc->Words.Word[i].word),Doc->Words.Word[i].coord);
					}
					if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
						return rc;
				}
			}
		}
		
		if(1){
			switch(db->DBType){
				case UDM_DB_PGSQL:
					rc=UdmSQLQuery(db,NULL,"END WORK");
					break;
				case UDM_DB_ORACLE7:
				case UDM_DB_ORACLE8:
				case UDM_DB_SAPDB:
					rc=UdmSQLQuery(db,NULL,"COMMIT");
					db->commit_fl = 0;
					break;
				default:
#if (HAVE_IODBC || HAVE_UNIXODBC || HAVE_SOLID || HAVE_VIRT || HAVE_EASYSOFT || HAVE_SAPDB || HAVE_DB2)
					rc=UdmSQLQuery(db,NULL,"UDM_COMMIT");
#endif
					db->commit_fl = 0;
					break;
			}
			if(rc!=UDM_OK)return rc;
		}
	}
	return(UDM_OK);
}


static int StoreWordsSingle(UDM_AGENT * Indexer,UDM_DOCUMENT * Doc,UDM_DB *db){
	size_t	i;
	char	qbuf[256]="";
	time_t	stmp;
	int	rc=UDM_OK;
	int	url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
	
	stmp=time(NULL);
	
	/* Start transaction if supported */
	/* This is to make stuff faster   */

	if(1){
		switch(db->DBType){
			case UDM_DB_PGSQL:
				rc=UdmSQLQuery(db,NULL,"BEGIN WORK");
				break;
			case UDM_DB_MSSQL:
				rc=UdmSQLQuery(db,NULL,"BEGIN TRANSACTION");
				break;
			case UDM_DB_ORACLE7:
			case UDM_DB_ORACLE8:
			case UDM_DB_SAPDB:
				rc=UdmSQLQuery(db,NULL,"COMMIT");
				db->commit_fl = 1;
				break;
			case UDM_DB_IBASE:
				rc=UdmSQLQuery(db,NULL,"BEGIN");
				db->commit_fl = 1;
				break;
			default:
				rc=UDM_OK;
				db->commit_fl = 1;
				break;
		}
		if(rc!=UDM_OK)return rc;
	}
	
	/* Delete old words */
	if(db->DBMode==UDM_DBMODE_SINGLE){
		sprintf(qbuf,"DELETE FROM dict WHERE url_id=%d",url_id);
	}else
	if(db->DBMode==UDM_DBMODE_SINGLE_CRC){
		sprintf(qbuf,"DELETE FROM ndict WHERE url_id=%d",url_id);
	}
	
	if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
		return(UDM_ERROR);
	
	/* Insert new words */
	if(db->DBType==UDM_DB_MYSQL){
		if(Doc->Words.nwords){
			size_t nstored=0;
			
			while(nstored<Doc->Words.nwords){
				char * qb,*qe;
				size_t step=1024;
				size_t mlen=1024;

				qb=(char*)malloc(mlen);
				if(db->DBMode==UDM_DBMODE_SINGLE){
					strcpy(qb,"INSERT INTO dict (word,url_id,intag) VALUES ");
				}else
				if(db->DBMode==UDM_DBMODE_SINGLE_CRC){
					strcpy(qb,"INSERT INTO ndict (url_id,word_id,intag) VALUES ");
				}
				qe=qb+strlen(qb);

				for(i=nstored;i<Doc->Words.nwords;i++){
					size_t len=qe-qb;
					if(!Doc->Words.Word[i].coord) { nstored++; continue;}

				
					/* UDM_MAXWORDSIZE+100 should be enough */
					if((len+UDM_MAXWORDSIZE+100)>=mlen){
						mlen+=step;
						qb=(char*)realloc(qb,mlen);
						qe=qb+len;
					}
					
					if(i>nstored)*qe++=',';

					if(db->DBMode==UDM_DBMODE_SINGLE){
						*qe++='(';
						*qe++='\'';
						strcpy(qe,Doc->Words.Word[i].word);
						while(*qe)qe++;
						*qe++='\'';
						*qe++=',';
						qe+=sprintf(qe,"%d,%d",url_id,Doc->Words.Word[i].coord);
						*qe++=')';
						*qe='\0';
					}else
					if(db->DBMode==UDM_DBMODE_SINGLE_CRC){
						sprintf(qe,"(%d,%d,%d)",url_id,UdmStrCRC32(Doc->Words.Word[i].word),Doc->Words.Word[i].coord);
						qe=qe+strlen(qe);
					}else
					if(db->DBMode==UDM_DBMODE_BLOB){
						sprintf(qe,"('%s',%d,%d,%li)",Doc->Words.Word[i].word,url_id,Doc->Words.Word[i].coord,stmp);
						qe=qe+strlen(qe);
					}
					if(qe>qb+UDM_MAX_MULTI_INSERT_QSIZE)
						break;
				}
				nstored=i;
				rc=UdmSQLQuery(db,NULL,qb);
				free(qb);
				if(rc!=UDM_OK)return rc;
			}
		}
	}else{
		for(i=0;i<Doc->Words.nwords;i++){
			if(!Doc->Words.Word[i].coord)continue;
				
			if(db->DBMode==UDM_DBMODE_SINGLE){
				sprintf(qbuf,"INSERT INTO dict (url_id,word,intag) VALUES(%d,'%s',%d)",url_id,Doc->Words.Word[i].word,Doc->Words.Word[i].coord);
			}else{
				sprintf(qbuf,"INSERT INTO ndict (url_id,word_id,intag) VALUES(%d,%d,%d)",url_id,UdmStrCRC32(Doc->Words.Word[i].word),Doc->Words.Word[i].coord);
			}
			if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
				return rc;
		}
	}

	/* Commit */
	if(1){
		switch(db->DBType){
			case UDM_DB_PGSQL:
				rc=UdmSQLQuery(db,NULL,"END WORK");
				break;
			case UDM_DB_MSSQL:
				rc=UdmSQLQuery(db,NULL,"COMMIT");
				break;
			case UDM_DB_ORACLE7:
			case UDM_DB_ORACLE8:
			case UDM_DB_SAPDB:
				db->commit_fl = 0;
				rc=UdmSQLQuery(db,NULL,"COMMIT");
				break;
			case UDM_DB_IBASE:
				rc=UdmSQLQuery(db,NULL,"COMMIT");
				db->commit_fl = 1;
				break;
			default:
#if (HAVE_IODBC || HAVE_UNIXODBC || HAVE_SOLID || HAVE_VIRT || HAVE_EASYSOFT || HAVE_SAPDB || HAVE_DB2)
				rc=UdmSQLQuery(db,NULL,"UDM_COMMIT");
#endif
				db->commit_fl = 0;
				break;
		}
		if(rc!=UDM_OK)return rc;
	}
	return(UDM_OK);
}



static int UdmResStoreBlob(UDM_AGENT *Indexer,UDM_RESULT *Res,UDM_DB *db){
	size_t			tabnum,docnum,wnum;
	int			ts=(int)time(NULL);
	char			*blob;
	char			*qb,*e;
	int			*dellog;
	size_t			nbytes;
	UDM_URLCRDLISTLIST	CLst;
	int			rc=UDM_OK;
	
	if(!Res->num_rows)return UDM_OK;
	if(db->DBMode!=UDM_DBMODE_BLOB)return UDM_OK;
	
	bzero(&CLst,sizeof(CLst));
	/* Collect words from all documents */
	UdmResCollectWords(&Indexer->Conf->Indexed,&CLst);
	
	
	/* First of all insert delay log */
	
	switch(db->DBType){
	case UDM_DB_MYSQL:
		nbytes=Res->num_rows*sizeof(int);
		dellog=(int*)malloc(nbytes);
		for(docnum=0;docnum<Res->num_rows;docnum++){
			dellog[docnum]=UdmVarListFindInt(&Res->Doc[docnum].Sections,"ID",0);
		}
		blob=(char*)malloc(1+2*nbytes);
		qb=(char*)malloc(256+2*nbytes);
		UdmDBEscStr(db->DBType,blob,(char*)dellog,nbytes);
		UDM_FREE(dellog);
		sprintf(qb,"INSERT INTO dellog (tstamp,ndocs,docs) VALUES (%d,%d,'%s')",ts,Res->num_rows,blob);
		rc=UdmSQLQuery(db,NULL,qb);
		UDM_FREE(blob);
		UDM_FREE(qb);
		
	case UDM_DB_PGSQL:
	case UDM_DB_IBASE:
		nbytes=Res->num_rows*12+1; /* 0x80000000=-2147483648=11 chars */
		e=blob=(char*)malloc(nbytes);
		qb=(char*)malloc(nbytes+256);
		for(docnum=0;docnum<Res->num_rows;docnum++){
			if(e>blob)*e++=',';
			e+=sprintf(e,"%d",UdmVarListFindInt(&Res->Doc[docnum].Sections,"ID",0));
		}
		sprintf(qb,"INSERT INTO dellog (tstamp,ndocs,docs) VALUES (%d,%d,'{%s}')",ts,Res->num_rows,blob);
		rc=UdmSQLQuery(db,NULL,qb);
		free(blob);
		free(qb);
		break;
	default:
		sprintf(db->errstr,"Blob mode not supported by this DBType");
		db->errcode=1;
		rc=UDM_ERROR;
	}
	
	if(rc!=UDM_OK)goto ret;
	
	/* Insert words */
	for(tabnum=0;tabnum<CLst.nlists;tabnum++){
		UDM_URLCRDLIST	*Lst=&CLst.List[tabnum];
		int		table,crc32;
		
		if(!Lst->ncoords)continue;
		
		crc32=UdmStrCRC32(Lst->word);
		table=(unsigned)crc32 % db->numtables;
		
		switch(db->DBType){
		case UDM_DB_MYSQL:
			nbytes=sizeof(UDM_URL_CRD)*Lst->ncoords;
			blob=(char*)malloc(1+2*nbytes);
			qb=(char*)malloc(256+2*nbytes);
			UdmDBEscStr(db->DBType,blob,(char*)Lst->Coords,nbytes);
			sprintf(qb,"INSERT INTO bdict%02X (word,tstamp,ncoords,coords) VALUES ('%s',%d,%d,'%s')",table,Lst->word,ts,Lst->ncoords,blob);
			rc=UdmSQLQuery(db,NULL,qb);
			UDM_FREE(blob);
			UDM_FREE(qb);
			break;
		case UDM_DB_IBASE:
		case UDM_DB_PGSQL:
			nbytes=Lst->ncoords*sizeof(UDM_URL_CRD)*12+1;
			e=blob=(char*)malloc(nbytes);
			qb=(char*)malloc(256+nbytes);
			for(wnum=0;wnum<Lst->ncoords;wnum++){
				if(e!=blob)*e++=',';
				e+=sprintf(e,"%d,%d",Lst->Coords[wnum].url_id,Lst->Coords[wnum].coord);
			}
			if(Lst->ncoords<1024){
				sprintf(qb,"INSERT INTO bdict%02X (word,tstamp,ncoords,coords) VALUES ('%s',%d,%d,'{%s}')",table,Lst->word,ts,Lst->ncoords,blob);
				rc=UdmSQLQuery(db,NULL,qb);
			}else{
				printf("FIXME SKIP: too long: '%s' %d coords\n",Lst->word,Lst->ncoords);
			}
			free(blob);
			free(qb);
			break;
		}
		if(rc!=UDM_OK)goto ret;
	}
ret:
	UdmURLCRDListListFree(&CLst);
	return rc;
}


static int UdmStoreWords(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db){
	int	res;
	
	switch(db->DBMode){
		case UDM_DBMODE_BLOB:
			res=UDM_OK;
			break;
		case UDM_DBMODE_CACHE:
			res=UdmStoreWordsCache(Indexer,Doc,db);
			break;
		case UDM_DBMODE_MULTI:
		case UDM_DBMODE_MULTI_CRC:
			res=StoreWordsMulti(Indexer,Doc,db);
			break;
		case UDM_DBMODE_SINGLE:
		case UDM_DBMODE_SINGLE_CRC:
		default:
			res=StoreWordsSingle(Indexer,Doc,db);
			break;
	}
	return(res);
}


static int UdmDeleteAllFromDict(UDM_AGENT *Indexer,UDM_DB *db){
	char	qbuf[512];
	size_t	i,last=0;
	int	rc=UDM_OK;
	
	switch(db->DBMode){
	case UDM_DBMODE_MULTI:
		for(i=MINDICT;i<MAXDICT;i++){
			if(last!=DICTNUM(i)){
				if (db->DBSQL_TRUNCATE)
					sprintf(qbuf,"TRUNCATE TABLE dict%d",DICTNUM(i));
				else
					sprintf(qbuf,"DELETE FROM dict%d",DICTNUM(i));

				if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
					return rc;
				last=DICTNUM(i);
			}
		}
		break;
	case UDM_DBMODE_MULTI_CRC:
		for(i=MINDICT;i<MAXDICT;i++){
			if(last!=DICTNUM(i)){
				if (db->DBSQL_TRUNCATE)
					sprintf(qbuf,"TRUNCATE TABLE ndict%d",DICTNUM(i));
				else
					sprintf(qbuf,"DELETE FROM ndict%d",DICTNUM(i));
	
				if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
					return(UDM_ERROR);
				last=DICTNUM(i);
			}
		}
		break;
	case UDM_DBMODE_SINGLE_CRC:
		if (db->DBSQL_TRUNCATE)
			rc=UdmSQLQuery(db,NULL,"TRUNCATE TABLE ndict");
		else
			rc=UdmSQLQuery(db,NULL,"DELETE FROM ndict");
		break;
	case UDM_DBMODE_BLOB:
		for(i=0;i<db->numtables;i++){
			if (db->DBSQL_TRUNCATE)
				sprintf(qbuf,"TRUNCATE TABLE bdict%02X",i);
			else
				sprintf(qbuf,"DELETE FROM bdict%02X",i);
			if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
				return rc;
		}
		rc=UdmSQLQuery(db,NULL,"DELETE FROM dellog");
		break;
	default:
		if (db->DBSQL_TRUNCATE)
			rc=UdmSQLQuery(db,NULL,"TRUNCATE TABLE dict");
		else
			rc=UdmSQLQuery(db,NULL,"DELETE FROM dict");
		break;
	}
	return rc;
}


/***************** CrossWords *******************************/

static int UdmDeleteAllFromCrossDict(UDM_AGENT * Indexer,UDM_DB *db){
	char	qbuf[1024];
	char	table[64]="ncrossdict";
	int	crcmode=1;
	
	if((db->DBMode==UDM_DBMODE_SINGLE)||(db->DBMode==UDM_DBMODE_MULTI)||(db->DBMode==UDM_DBMODE_CACHE)){
		strcpy(table,"crossdict");
		crcmode=0;
	}
	sprintf(qbuf,"DELETE FROM %s",table);
	return UdmSQLQuery(db,NULL,qbuf);
}


static int UdmDeleteCrossWordFromURL(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db){
	char	qbuf[1024];
	char	table[64]="ncrossdict";
	int	crcmode=1;
	int	url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
	int	referrer_id=UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0);
	int	rc=UDM_OK;
	
	if((db->DBMode==UDM_DBMODE_SINGLE)||(db->DBMode==UDM_DBMODE_MULTI)||(db->DBMode==UDM_DBMODE_CACHE)){
		strcpy(table,"crossdict");
		crcmode=0;
	}
	if(url_id){
		sprintf(qbuf,"DELETE FROM %s WHERE url_id=%d",table,url_id);
		if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
			return rc;
	}
	if(referrer_id){
		sprintf(qbuf,"DELETE FROM %s WHERE ref_id=%d",table,referrer_id);
		rc=UdmSQLQuery(db,NULL,qbuf);
	}
	return rc;
}


static int UdmStoreCrossWords(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db){
	UDM_DOCUMENT	U;
	size_t		i;
	char		qbuf[1024];
	char		table[64]="ncrossdict";
	int		crcmode=1;
	const char	*lasturl="scrap";
	int		referrer=UdmVarListFindInt(&Doc->Sections,"ID",0);
	int		childid=0;
	int		rc=UDM_OK;
	
	UdmDocInit(&U);
	UdmVarListReplaceInt(&Doc->Sections,"Referrer-ID",referrer);
	if(UDM_OK!=(rc=UdmDeleteCrossWordFromURL(Indexer,&U,db))){
		UdmDocFree(&U);
		return rc;
	}
	
	if(Doc->CrossWords.ncrosswords==0) {
		UdmDocFree(&U);
		return rc;
	}
	
	if((db->DBMode==UDM_DBMODE_SINGLE)||(db->DBMode==UDM_DBMODE_MULTI)||(db->DBMode==UDM_DBMODE_CACHE)){
		strcpy(table,"crossdict");
		crcmode=0;
	}
	
	for(i=0;i<Doc->CrossWords.ncrosswords;i++){
		if(!Doc->CrossWords.CrossWord[i].weight)continue;
		if(strcmp(lasturl,Doc->CrossWords.CrossWord[i].url)){
			UdmVarListReplaceStr(&U.Sections,"URL",Doc->CrossWords.CrossWord[i].url);
			if(UDM_OK!=(rc=UdmFindURL(Indexer,&U,db))){
				UdmDocFree(&U);
				return rc;
			}
			childid=UdmVarListFindInt(&U.Sections,"ID",0);
			lasturl=Doc->CrossWords.CrossWord[i].url;
		}
		Doc->CrossWords.CrossWord[i].referree_id=childid;
	}
	
	/* Begin transacttion/lock */
	if(1){
		switch(db->DBType){
			case UDM_DB_MYSQL:
				sprintf(qbuf,"LOCK TABLES %s WRITE",table);
				rc=UdmSQLQuery(db,NULL,qbuf);
				break;
			case UDM_DB_PGSQL:
				rc=UdmSQLQuery(db,NULL,"BEGIN WORK");
				break;
			case UDM_DB_ORACLE7:
			case UDM_DB_ORACLE8:
			case UDM_DB_SAPDB:
				db->commit_fl = 1;
				rc=UdmSQLQuery(db,NULL,"COMMIT");
				break;
			default:
				db->commit_fl = 1;
				break;
		}
		if(rc!=UDM_OK){
			UdmDocFree(&U);
			return rc;
		}
	}
	
	/* Insert new words */
	for(i=0;i<Doc->CrossWords.ncrosswords;i++){
		if(Doc->CrossWords.CrossWord[i].weight && Doc->CrossWords.CrossWord[i].referree_id){
			int weight=UDM_WRDCOORD(Doc->CrossWords.CrossWord[i].pos,Doc->CrossWords.CrossWord[i].weight);
			if(crcmode){
				sprintf(qbuf,"INSERT INTO %s (ref_id,url_id,word_id,intag) VALUES(%d,%d,%d,%d)",table,referrer,Doc->CrossWords.CrossWord[i].referree_id,UdmStrCRC32(Doc->CrossWords.CrossWord[i].word),weight);
			}else{
				sprintf(qbuf,"INSERT INTO %s (ref_id,url_id,word,intag) VALUES(%d,%d,'%s',%d)",table,referrer,Doc->CrossWords.CrossWord[i].referree_id,Doc->CrossWords.CrossWord[i].word,weight);
			}
			if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf))){
				UdmDocFree(&U);
				return rc;
			}
		}
	}

	/* COMMIT/UNLOCK */
	if(1){
		switch(db->DBType){
			case UDM_DB_MYSQL:
				rc=UdmSQLQuery(db,NULL,"UNLOCK TABLES");
				break;
			case UDM_DB_PGSQL:
				rc=UdmSQLQuery(db,NULL,"END WORK");
				break;
			case UDM_DB_ORACLE7:
			case UDM_DB_ORACLE8:
			case UDM_DB_SAPDB:
				db->commit_fl = 0;
				rc=UdmSQLQuery(db,NULL,"COMMIT");
				break;
			default:
#if (HAVE_IODBC || HAVE_UNIXODBC || HAVE_SOLID || HAVE_VIRT || HAVE_EASYSOFT || HAVE_SAPDB || HAVE_DB2)
				rc=UdmSQLQuery(db,NULL,"UDM_COMMIT");
#endif
				db->commit_fl = 0;
				break;
		}
	}
	UdmDocFree(&U);
	return rc;
}




/************************ URLs ***********************************/


static int UdmAddURL(UDM_AGENT *Indexer,UDM_DOCUMENT * Doc,UDM_DB *db){
	char		e_url[UDM_URLSIZE*4]="";
	int		next_url_id=0;
	char		qbuf[UDM_URLSIZE*4]="AddURL";
	UDM_SQLRES	SQLRes;
	const char	*url=UdmVarListFindStr(&Doc->Sections,"URL","");
	int		url_seed;
	int		use_crc32_url_id=!strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars,"UseCRC32URLId","no"),"yes");
	int		rc=UDM_OK;
	
	url_seed = UdmStrCRC32(url) & 0xFF;
	
	/* Escape URL string */
	UdmDBEscStr(db->DBType,e_url,url,strlen(url));
	
	if(use_crc32_url_id){
		/* Auto generation of rec_id */
		/* using CRC32 algorythm     */
		int rec_id=UdmStrCRC32(url);
		
		snprintf(qbuf,sizeof(qbuf)-1,"INSERT INTO url (rec_id,url,referrer,hops,crc32,next_index_time,status,tag,category,seed,bad_since_time) VALUES (%d,'%s',%d,%d,0,%d,0,'%s','%s',%d,%d)",
			rec_id,
			e_url,
			UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
			UdmVarListFindInt(&Doc->Sections,"Hops",0),
			(int)time(NULL),
			UdmVarListFindStr(&Doc->Sections,"Tag",""),
			UdmVarListFindStr(&Doc->Sections,"Category",""),
			url_seed, (int)time(NULL) );
	}else{
		/* Use dabatase generated rec_id */
		/* It depends on used DBType     */
		switch(db->DBType){
		case UDM_DB_MSQL:
			/* miniSQL has _seq as autoincrement value */
			if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,"SELECT _seq FROM url")))
				return rc;
			next_url_id=atoi(UdmSQLValue(&SQLRes,0,0));
			UdmSQLFree(&SQLRes);
			snprintf(qbuf,sizeof(qbuf)-1,"INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,tag,category,seed,bad_since_time) VALUES ('%s',%d,%d,%d,0,%d,0,'%s','%s',%d,%d)",
				e_url,
				UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
				UdmVarListFindInt(&Doc->Sections,"Hops",0),
				next_url_id,
				(int)time(NULL),
				UdmVarListFindStr(&Doc->Sections,"Tag",""),
				UdmVarListFindStr(&Doc->Sections,"Category",""),
				url_seed, (int)time(NULL) );
			break;

		case UDM_DB_SOLID:
		case UDM_DB_ORACLE7:
		case UDM_DB_ORACLE8:
		case UDM_DB_SAPDB:
			/* FIXME: Dirty hack for stupid too smart databases */
			if (strlen(e_url)>UDM_URLSIZE)e_url[UDM_URLSIZE]=0;
			/* Use sequence next_url_id.nextval */
			snprintf(qbuf,sizeof(qbuf)-1,"INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,tag,category,seed,bad_since_time) VALUES ('%s',%d,%d,next_url_id.nextval,0,%d,0,'%s','%s',%d,%d)",
				e_url,
				UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
				UdmVarListFindInt(&Doc->Sections,"Hops",0),
				(int)time(NULL),
				UdmVarListFindStr(&Doc->Sections,"Tag",""),
				UdmVarListFindStr(&Doc->Sections,"Category",""),
				url_seed, (int)time(NULL) );
			break;
		case UDM_DB_IBASE:
			snprintf(qbuf,sizeof(qbuf)-1,"INSERT INTO url (url,referrer,hops,rec_id,crc32,next_index_time,status,tag,category,seed,bad_since_time) VALUES ('%s',%d,%d,GEN_ID(rec_id_GEN,1),0,%d,0,'%s','%s',%d,%d)",
				e_url,
				UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
				UdmVarListFindInt(&Doc->Sections,"Hops",0),
				(int)time(NULL),
				UdmVarListFindStr(&Doc->Sections,"Tag",""),
				UdmVarListFindStr(&Doc->Sections,"Category",""),
				url_seed, (int)time(NULL) );
			break;
		case UDM_DB_MYSQL:
			/* MySQL generates itself */
		default:	
			snprintf(qbuf,sizeof(qbuf)-1,"INSERT INTO url (url,referrer,hops,crc32,next_index_time,status,tag,category,seed,bad_since_time) VALUES ('%s',%d,%d,0,%d,0,'%s','%s',%d,%d)",
				e_url,
				UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0),
				UdmVarListFindInt(&Doc->Sections,"Hops",0),
				(int)time(NULL),
				UdmVarListFindStr(&Doc->Sections,"Tag",""),
				UdmVarListFindStr(&Doc->Sections,"Category",""),
				url_seed, (int)time(NULL) );
		}
	}

	/* Exec INSERT now */
	rc = UdmSQLQuery(db, NULL, qbuf); /* do not check return code - inserting dublicate urls is normal */

	return UDM_OK;
}


static int UdmDeleteURL(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db){
	char	qbuf[128];
	int	rc;
	int	url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
	int	use_crosswords=!strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars,"CrossWords","no"),"yes");
	
	if(use_crosswords&&db->DBMode!=UDM_DBMODE_CACHE)
		if(UDM_OK!=(rc=UdmDeleteCrossWordFromURL(Indexer,Doc,db)))return(rc);
	
	if(UDM_OK!=(rc=UdmDeleteWordFromURL(Indexer,Doc,db)))return(rc);
	
	sprintf(qbuf,"DELETE FROM url WHERE rec_id=%d",url_id);
	if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
	
	sprintf(qbuf,"DELETE FROM urlinfo WHERE url_id=%d",url_id);
	if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))return rc;
	
	UdmStoreDeleteDoc(Indexer, Doc);
	return rc;
}

static int UdmMarkForReindex(UDM_AGENT *Indexer,UDM_DB *db){
	char		qbuf[1024];
	const char	*where=BuildWhere(Indexer->Conf,db);
	
	snprintf(qbuf,sizeof(qbuf),"UPDATE url SET next_index_time=%d WHERE rec_id<>0 %s %s",(int)time(NULL),where[0]?"AND":"",where);
	return UdmSQLQuery(db,NULL,qbuf);
}

static int UdmDeleteBadHrefs(UDM_AGENT *Indexer, int url_id, UDM_DB *db) {
	UDM_DOCUMENT	rDoc;
	UDM_SQLRES	SQLRes;
	char		q[128];
	size_t		i;
	size_t		nrows;
	int		rc=UDM_OK;
	
	sprintf(q, "SELECT rec_id FROM url WHERE status >=300 AND status<>304 AND referrer=%d AND bad_since_time<%d",
		url_id, (int)time(NULL) - Indexer->Conf->bad_since_time);
	if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,q)))return rc;
	
	nrows = UdmSQLNumRows(&SQLRes);
	
	UdmDocInit(&rDoc);
	for(i = 0; i < nrows ; i++) {
		UdmVarListReplaceStr(&rDoc.Sections,"ID", UdmSQLValue(&SQLRes,i,0));
		if(UDM_OK!=(rc=UdmDeleteURL(Indexer, &rDoc, db)))
			break;
	}
	UdmDocFree(&rDoc);
	UdmSQLFree(&SQLRes);
	return rc;
}

static int UdmUpdateUrl(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db){
	char	qbuf[256]="";
	int	url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
	int	status=UdmVarListFindInt(&Doc->Sections,"Status",0);
	int	prevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0);
	int	next_index_time=UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Next-Index-Time",""));
	int	res;
	
#ifdef HAVE_ORACLE8
	if(db->DBDriver==UDM_DB_ORACLE8){
	  if ( (prevStatus != status) && (status > 300) && (status != 304) ) {
		sprintf(qbuf,"UPDATE url SET status=:1,next_index_time=:2,bad_since_time=:4 WHERE rec_id=:3");
		param_init(db, 1, 2, 3, 4);
		param_add(db, status, next_index_time, url_id, time(NULL));
	  } else {
		sprintf(qbuf,"UPDATE url SET status=:1,next_index_time=:2 WHERE rec_id=:3");
		param_init(db, 1, 2, 3, 0);
		param_add(db, status, next_index_time, url_id, 0);
	  }
	}
#endif
	if(!qbuf[0]) {
	  if ( (prevStatus != status) && (status > 300) && (status != 304) )
		sprintf(qbuf, "UPDATE url SET status=%d,next_index_time=%d,bad_since_time=%d WHERE rec_id=%d",
			status, next_index_time, (int)time(NULL), url_id);
	  else
		sprintf(qbuf,"UPDATE url SET status=%d,next_index_time=%d WHERE rec_id=%d",status,next_index_time,url_id);
	}
	
	if(UDM_OK!=(res=UdmSQLQuery(db,NULL,qbuf)))return res;
	
	/* remove all old broken hrefs from this document to avoid broken link collecting */
	return UdmDeleteBadHrefs(Indexer, url_id, db);
}

static int UdmRegisterChild(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db)
{
	char	qbuf[1024];
	int	url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
	int	parent_id=UdmVarListFindInt(&Doc->Sections,"Parent-ID",0);
	
	snprintf(qbuf,sizeof(qbuf),"insert into thread values(%d,%d)",parent_id,url_id);
	return UdmSQLQuery(db,NULL,qbuf);
}

static int UdmDeleteAllFromThread(UDM_AGENT *Indexer,UDM_DB *db){
	return UdmSQLQuery(db,NULL,"DELETE FROM thread");
}



static int UdmLongUpdateURL(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db){
	int		rc=UDM_OK;
	int		i;
	char		*qbuf;
	char		*arg;
	char		qsmall[128];
	size_t		len=0;
	UDM_VAR		*var;
	const char	*charset;
	int		url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
	int	        status = UdmVarListFindInt(&Doc->Sections, "Status", 0);
	int	        prevStatus = UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0);
	
	charset = UdmVarListFindStr(&Doc->Sections,"Charset","iso-8859-1");
	charset = UdmCharsetCanonicalName(charset);
	
	if((var=UdmVarListFind(&Doc->Sections,"Content-Language"))){
		len=strlen(var->val);
		for(i = 0; i < len; i++) {
			var->val[i] = tolower(var->val[i]);
		}
	}
	
	if ( (prevStatus != status) && (status > 300) && (status != 304) )
	  sprintf(qsmall, ", bad_since_time=%d", (int)time(NULL));
	else qsmall[0] = '\0';
	    
	qbuf=(char*)malloc(1024);
	
snprintf(qbuf, 1023, "\
UPDATE url SET \
status=%d,last_mod_time=%li,\
next_index_time=%li,\
tag='%s',content_type='%s',docsize=%d,\
crc32=%d,lang='%s',category='%s', charset='%s'%s \
WHERE rec_id=%d",
	UdmVarListFindInt(&Doc->Sections,"Status",0),
	UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Last-Modified","")),
	UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Next-Index-Time","")),
	UdmVarListFindStr(&Doc->Sections,"Tag",""),
	UdmVarListFindStr(&Doc->Sections,"Content-Type",""),
	UdmVarListFindInt(&Doc->Sections,"Content-Length",0),
	UdmVarListFindInt(&Doc->Sections,"crc32",0),
	UdmVarListFindStr(&Doc->Sections,"Content-Language",""),
	UdmVarListFindStr(&Doc->Sections,"Category",""),
	(charset == NULL)?"":charset,
	qsmall,
	url_id);
	
	rc=UdmSQLQuery(db,NULL,qbuf);
	free(qbuf);
	if(rc!=UDM_OK)return rc;
	
	/* remove all old broken hrefs from this document to avoid broken link collecting */
	if(UDM_OK!=(rc=UdmDeleteBadHrefs(Indexer, url_id, db)))return rc;
	
	sprintf(qsmall,"DELETE FROM urlinfo WHERE url_id=%d",url_id);
	if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qsmall)))return rc;
	
	if(!Doc->Sections.nvars) return UDM_OK;
	
	len=0;
	for(i=0;i<Doc->Sections.nvars;i++){
		if(len<Doc->Sections.Var[i].curlen)
			len=Doc->Sections.Var[i].curlen;
	}
	if(!len)return UDM_OK;
	
	qbuf=(char*)malloc(len+128);
	arg=(char*)malloc(len+128);
	
	for(i=0;i<Doc->Sections.nvars;i++){
		UDM_VAR *Sec=&Doc->Sections.Var[i];
		if(Sec->curlen && Sec->val && Sec->name && Sec->section){
			arg=UdmDBEscStr(db->DBType,arg,Sec->val,strlen(Sec->val));
			sprintf(qbuf,"INSERT INTO urlinfo (url_id,sname,sval) VALUES (%d,'%s','%s')",
				url_id,Sec->name,arg);
			if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))break;
		}
	}
	free(qbuf);
	free(arg);
	return rc;
}


static int UdmDeleteAllFromUrl(UDM_AGENT *Indexer,UDM_DB *db){
	int	rc;
	
	if(db->DBSQL_TRUNCATE)
		rc=UdmSQLQuery(db,NULL,"TRUNCATE TABLE url");
	else
		rc=UdmSQLQuery(db,NULL,"DELETE FROM url");
	
	if(rc!=UDM_OK)return rc;
	
	if(db->DBSQL_TRUNCATE)
		rc=UdmSQLQuery(db,NULL,"TRUNCATE TABLE urlinfo");
		
	else
		rc=UdmSQLQuery(db,NULL,"DELETE FROM urlinfo");
	
	return rc;
}



/************************ Clones stuff ***************************/
static int UdmFindOrigin(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db){
	size_t		i=0;
	char		qbuf[256]="";
	UDM_SQLRES	SQLRes;
	int		origin_id=0;
	int		crc32=UdmVarListFindInt(&Doc->Sections,"crc32",0);
	int		rc;
	
	if (crc32==0)return UDM_OK;
	
	if (db->DBSQL_IN)
		sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=%d AND status IN (200,304,206)",crc32);
	else
		sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=%d AND (status=200 OR status=304 OR status=206)",crc32);
	
	if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,qbuf)))
		return rc;
	
	for(i=0;i<UdmSQLNumRows(&SQLRes);i++){
		const char *o;
		if((o=UdmSQLValue(&SQLRes,i,0)))
			if((!origin_id)||(origin_id>atoi(o)))
				origin_id=atoi(o);
	}
	UdmSQLFree(&SQLRes);
	UdmVarListReplaceInt(&Doc->Sections,"Origin-ID",origin_id);
	return(UDM_OK);
}

static int  UdmUpdateClone(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc,UDM_DB *db){
	char	*qbuf;
	int	rc;
	
	/* This should be enough */
	qbuf=(char*)malloc(1024);
	snprintf(qbuf, 1023, "UPDATE url SET crc32=%d,status=%d,content_type='%s',last_mod_time=%li,next_index_time=%li,docsize=%d WHERE rec_id=%d",
		UdmVarListFindInt(&Doc->Sections,"crc32",0),
		UdmVarListFindInt(&Doc->Sections,"Status",0),
		UdmVarListFindStr(&Doc->Sections,"Content-Type",""),
		UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Last-Modified","")),
		UdmHttpDate2Time_t(UdmVarListFindStr(&Doc->Sections,"Next-Index-Time","")),
		UdmVarListFindInt(&Doc->Sections,"Content-Length",0),
		UdmVarListFindInt(&Doc->Sections,"ID",0));
	
	rc=UdmSQLQuery(db,NULL,qbuf);
	free(qbuf);
	return rc;
}


/************** Get Target to be indexed ***********************/

static int UdmTargets(UDM_AGENT *Indexer,UDM_RESULT *Res,UDM_DB *db){
	char		sortstr[64]="";
	char		updstr[64]="";
	char		lmtstr[64]="";
	size_t		i=0;
	size_t		url_num=URL_SELECT_CACHE;
	UDM_SQLRES 	SQLRes;
	char		smallbuf[128];
	int		rc=UDM_OK;
	
	UdmResultFree(&Indexer->Conf->Targets);
	
	if(1){
		switch(db->DBType){
			case UDM_DB_MYSQL:
				rc=UdmSQLQuery(db,NULL,"LOCK TABLES url WRITE");
				break;
			case UDM_DB_PGSQL:
				rc=UdmSQLQuery(db,NULL,"BEGIN WORK");
				rc=UdmSQLQuery(db,NULL,"LOCK url");
				break;
			case UDM_DB_ORACLE7:
			case UDM_DB_ORACLE8:
				sprintf(updstr, " FOR UPDATE ");
#if HAVE_ORACLE8
				if(db->DBDriver==UDM_DB_ORACLE8){
					sprintf(lmtstr, " AND ROWNUM <=%d",url_num); 
				}
#endif
				if(!lmtstr[0])
					sprintf(lmtstr, " AND ROWNUM <=%d", url_num); 
				break;
			case UDM_DB_SAPDB:
				sprintf(updstr, " WITH LOCK ");
				strcpy(lmtstr, "");
				break;
			default:
				break;
		}
		if(rc!=UDM_OK)return rc;
	}
	
	if(db->TargetsQuery == NULL) {
		const char	*where=BuildWhere(Indexer->Conf,db);
		char		*qbuf=NULL;
		
		if ( (qbuf = (char*)malloc(512 + 4 * UDM_URLSIZE)) == NULL) return UDM_ERROR;
		qbuf[0]='\0';
		
		sprintf(sortstr, " ORDER BY %s", (Indexer->flags&UDM_FLAG_SORT_EXPIRED) ? "seed, next_index_time" : "seed");
		if (Indexer->flags & UDM_FLAG_SORT_HOPS) {
			if (strlen(sortstr) == 0) {
				sprintf(sortstr," ORDER BY hops ASC ");
			} else {
				strcat(sortstr,", hops ASC ");
			}
		}
		
		switch(db->DBType){
			case UDM_DB_MYSQL:
				sprintf(qbuf,"SELECT url,rec_id,docsize,status,hops,crc32,last_mod_time,seed FROM url WHERE next_index_time<=unix_timestamp() %s %s %s LIMIT %d",where[0]?"AND":"",where,sortstr,url_num);
				break;
			case UDM_DB_PGSQL:
				sprintf(qbuf,"SELECT url,rec_id,docsize,status,hops,crc32,last_mod_time,seed FROM url WHERE next_index_time<=('now'::abstime)::int4 %s %s %s LIMIT %d",where[0]?"AND":"",where,sortstr,url_num);
				break;
			default:
			
				if(db->DBSQL_LIMIT){
					sprintf(qbuf,"SELECT url,rec_id,docsize,status,hops,crc32,last_mod_time,seed FROM url WHERE next_index_time<=%d %s %s %s LIMIT %d",
						(int)time(NULL),where[0]?"AND":"",where,sortstr,url_num);
				}else{
					db->res_limit=url_num;
/*
#if HAVE_ORACLE8
					if(db->DBDriver==UDM_DB_ORACLE8){
						param_init(db, 1, 2, 0, 0);
						param_add(db, (int)time(NULL), url_num, 0, 0);
						sprintf(qbuf,"SELECT url,rec_id,docsize,status,hops,crc32,last_mod_time FROM url WHERE next_index_time<=:1 %s %s%s%s%s",
							where[0]?"AND":"",where,lmtstr,sortstr,updstr);
					}
#endif
*/
					if(!qbuf[0])
						sprintf(qbuf,"SELECT url,rec_id,docsize,status,hops,crc32,last_mod_time FROM url WHERE next_index_time<=%d %s %s%s%s%s",
							(int)time(NULL),where[0]?"AND":"",where,lmtstr,sortstr,updstr);
				}
		}
		db->TargetsQuery = strdup(qbuf);
		free(qbuf);
	}
	
	if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLRes,db->TargetsQuery)))return rc;
	
	switch(db->DBType){
		case UDM_DB_MYSQL:
		case UDM_DB_PGSQL:
			/* 
			   MySQL and PgSQL support now() so we can
			   keep the query string to reuse it later
			   without having to rebuild it every time
			*/
			break;
		default:
			UDM_FREE(db->TargetsQuery);
	}
	
	Indexer->Conf->Targets.num_rows=UdmSQLNumRows(&SQLRes);
	
	if(!Indexer->Conf->Targets.num_rows){
		UdmSQLFree(&SQLRes);
		goto unlock;
	}
	
	Indexer->Conf->Targets.Doc = (UDM_DOCUMENT*)UdmXmalloc(sizeof(UDM_DOCUMENT)*(Indexer->Conf->Targets.num_rows));
	
	for(i=0;i<Indexer->Conf->Targets.num_rows;i++){
		char		buf[64]="";
		time_t		last_mod_time;
		UDM_DOCUMENT	*Doc=&Indexer->Conf->Targets.Doc[i];
		
		UdmVarListAddStr(&Doc->Sections,"URL",UdmSQLValue(&SQLRes,i,0));
		UdmVarListAddInt(&Doc->Sections,"ID",atoi(UdmSQLValue(&SQLRes,i,1)));
		UdmVarListAddInt(&Doc->Sections,"Status",atoi(UdmSQLValue(&SQLRes,i,3)));
		UdmVarListAddInt(&Doc->Sections,"PrevStatus",atoi(UdmSQLValue(&SQLRes,i,3)));
		UdmVarListAddInt(&Doc->Sections,"Hops",atoi(UdmSQLValue(&SQLRes,i,4)));
		UdmVarListAddInt(&Doc->Sections,"crc32",atoi(UdmSQLValue(&SQLRes,i,5)));
		last_mod_time = (time_t) atol(UdmSQLValue(&SQLRes,i,6));
		UdmTime_t2HttpStr(last_mod_time, buf);
		if (strlen(buf) > 0) {
		  UdmVarListAddStr(&Doc->Sections,"Last-Modified",buf);
		}
	}
	UdmSQLFree(&SQLRes);
	
	
	if (db->DBSQL_IN) {
		char	*urlin=NULL;
		char	*qbuf=NULL;
		
		if ( (qbuf = (char*)realloc(qbuf, 512 + 4 * UDM_URLSIZE + 35 * Indexer->Conf->Targets.num_rows)) == NULL) {
			UDM_FREE(qbuf);
			return UDM_ERROR;
		}
		
		if ( (urlin = (char*)malloc(35 * Indexer->Conf->Targets.num_rows)) == NULL) {
			UDM_FREE(qbuf);
			return UDM_ERROR;
		}
		urlin[0]=0;
		
		for(i=0;i<Indexer->Conf->Targets.num_rows;i++){
			UDM_DOCUMENT	*Doc=&Indexer->Conf->Targets.Doc[i];
			int		url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
			
			if(urlin[0])strcat(urlin,",");
			sprintf(urlin+strlen(urlin),"%d",url_id);
		}
		sprintf(qbuf,"UPDATE url SET next_index_time=%d WHERE rec_id in (%s)",(int)(time(NULL)+URL_LOCK_TIME),urlin);
		free(urlin);
		rc=UdmSQLQuery(db,NULL,qbuf);
		free(qbuf);
		if(rc!=UDM_OK)return rc;
	}else{
		for(i=0;i<Indexer->Conf->Targets.num_rows;i++){
			UDM_DOCUMENT	*Doc=&Indexer->Conf->Targets.Doc[i];
			int		url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
			
			sprintf(smallbuf,"UPDATE url SET next_index_time=%d WHERE rec_id=%d",(int)(time(NULL)+URL_LOCK_TIME),url_id);
			if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,smallbuf)))return rc;
		}
	}
	
	
unlock:
	if(1){
		switch(db->DBType){
			case UDM_DB_MYSQL:
				rc=UdmSQLQuery(db,NULL,"UNLOCK TABLES");
				break;
			case UDM_DB_PGSQL:
				rc=UdmSQLQuery(db,NULL,"END WORK");
				break;
			default:
				break;
		}
	}
	return rc;
}


/************************************************************/
/* Misc functions                                           */
/************************************************************/

static int UdmGetDocCount(UDM_AGENT * Indexer,UDM_DB *db){
	char		qbuf[200]="";
	UDM_SQLRES	SQLres;
	int		rc;
	
	Indexer->doccount=0;
	sprintf(qbuf,NDOCS_QUERY);
	if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
	
	if(UdmSQLNumRows(&SQLres)){
		const char * s;
		s=UdmSQLValue(&SQLres,0,0);
		if(s)Indexer->doccount=atoi(s);
	}
	UdmSQLFree(&SQLres);
	return(UDM_OK);
}


int UdmStatActionSQL(UDM_AGENT *Indexer,UDM_STATLIST *Stats,UDM_DB *db){
	size_t		i,j;
	char		qbuf[2048];
	UDM_SQLRES	SQLres;
	int		have_group=db->DBSQL_GROUP;
	const char	*where;
	int		rc=UDM_OK;
	
	if(db->DBType==UDM_DB_IBASE)
		have_group=0;
	
	where=BuildWhere(Indexer->Conf,db);
	bzero(Stats,sizeof(Stats[0]));

	if(have_group){
		
		switch(db->DBType){
		
			case UDM_DB_MYSQL:
				snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,sum(next_index_time<=%d),count(*) FROM url WHERE rec_id<>0 %s %s GROUP BY status",
					(int)time(NULL),where[0]?"AND":"",where);
				break;
		
			case UDM_DB_PGSQL:
			case UDM_DB_MSSQL:
			case UDM_DB_DB2:
				snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,sum(case when next_index_time<=%d then 1 else 0 end),count(*) FROM url WHERE rec_id<>0 %s %s GROUP BY status",
					(int)time(NULL),where[0]?"AND":"",where);
				break;

			case UDM_DB_ORACLE7:
			case UDM_DB_ORACLE8:
			case UDM_DB_SAPDB:
				snprintf(qbuf,sizeof(qbuf)-1,"SELECT status, SUM(DECODE(SIGN(%d-next_index_time),-1,0,1,1)), count(*) FROM url WHERE rec_id<>0 %s %s GROUP BY status",(int)time(NULL),
					where[0]?"AND":"",where);
				break;
		}

		if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
		
		if((j=UdmSQLNumRows(&SQLres))){
			for(i=0;i<j;i++){
				UDM_STAT	*S;
				
				Stats->Stat=(UDM_STAT*)realloc(Stats->Stat,(Stats->nstats+1)*sizeof(Stats->Stat[0]));
				S=&Stats->Stat[Stats->nstats];
				S->status=atoi(UdmSQLValue(&SQLres,i,0));
				S->expired=atoi(UdmSQLValue(&SQLres,i,1));
				S->total=atoi(UdmSQLValue(&SQLres,i,2));
				Stats->nstats++;
			}
		}
		UdmSQLFree(&SQLres);
	
	}else{
/*	
	FIXME: learn how to get it from SOLID and IBASE
	(HAVE_IBASE||HAVE_MSQL || HAVE_SOLID || HAVE_VIRT )
*/
		int	n=time(NULL);
		
		snprintf(qbuf,sizeof(qbuf)-1,"SELECT status,next_index_time FROM url WHERE rec_id>0 %s %s",
			where[0]?"AND":"",where);

		if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
		
		for(i=0;i<UdmSQLNumRows(&SQLres);i++){
			for(j=0;j<Stats->nstats;j++){
				if(Stats->Stat[j].status==atoi(UdmSQLValue(&SQLres,i,0))){
					if(atoi(UdmSQLValue(&SQLres,i,1))<=n)
						Stats->Stat[j].expired++;
					Stats->Stat[j].total++;
					break;
				}
			}
			if(j==Stats->nstats){
				Stats->Stat=(UDM_STAT *)realloc(Stats->Stat,sizeof(UDM_STAT)*(Stats->nstats+1));
				Stats->Stat[j].status=atoi(UdmSQLValue(&SQLres,i,0));
				Stats->Stat[j].expired=0;
				if(atoi(UdmSQLValue(&SQLres,i,1))<=n)
					Stats->Stat[j].expired++;
				Stats->Stat[j].total=1;
				Stats->nstats++;
			}
		}
		UdmSQLFree(&SQLres);
	}
	return rc;
}


static int UdmGetReferers(UDM_AGENT *Indexer,UDM_DB *db){
	size_t		i,j;
	char		qbuf[2048];
	UDM_SQLRES	SQLres;
	const char	*where=BuildWhere(Indexer->Conf,db);
	int		rc;
	
	snprintf(qbuf,sizeof(qbuf),"SELECT url.status,url2.url,url.url FROM url,url url2 WHERE url.referrer=url2.rec_id %s %s",
		where[0]?"AND":"",where);
	
	if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))return rc;
	
	j=UdmSQLNumRows(&SQLres);
	for(i=0;i<j;i++){
		if(Indexer->Conf->RefInfo)Indexer->Conf->RefInfo(
			atoi(UdmSQLValue(&SQLres,i,0)),
			UdmSQLValue(&SQLres,i,2),
			UdmSQLValue(&SQLres,i,1)
		);
	}
	UdmSQLFree(&SQLres);
	return rc;
}


int UdmClearDBSQL(UDM_AGENT *Indexer,UDM_DB *db){
	size_t		i,j;
	int		rc;
	char		qbuf[1024*4];
	const char*	where=BuildWhere(Indexer->Conf,db);
	int		use_newsext=!strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars,"NewsExtensions","no"),"yes");
	int		use_crosswords=!strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars,"CrossWords","no"),"yes");
	
	if(!where[0]){

		if(use_newsext){
			if((UDM_OK!=(rc=UdmDeleteAllFromThread(Indexer,db))))return(rc);
		}
		
		if(db->DBMode==UDM_DBMODE_CACHE){
			UdmClearCacheTree(Indexer->Conf);
		}

		if(use_crosswords&&db->DBMode!=UDM_DBMODE_CACHE){
			if((UDM_OK!=(rc=UdmDeleteAllFromCrossDict(Indexer,db))))return(rc);
		}
		if((UDM_OK!=(rc=UdmDeleteAllFromDict(Indexer,db))))return(rc);
		if((UDM_OK!=(rc=UdmDeleteAllFromUrl(Indexer,db))))return(rc);
	}else{
		j=0;
		while(1){
			int 		last=0;
			char 		urlin[1024*3]="";
			char		limit[100]="";
			UDM_SQLRES	SQLres;
			
			if(db->DBSQL_LIMIT){
				sprintf(limit," LIMIT %d",URL_DELETE_CACHE);
			}
			sprintf(qbuf,"SELECT rec_id FROM url WHERE rec_id<>0 AND %s %s",where,limit);
			
			if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
				return rc;
			
			if(UdmSQLNumRows(&SQLres)){
				UDM_DOCUMENT Doc;
				
				bzero(&Doc,sizeof(Doc));
				if(db->DBMode==UDM_DBMODE_CACHE){
					for(i=0;i<UdmSQLNumRows(&SQLres);i++){
						UdmVarListReplaceInt(&Doc.Sections,"ID",atoi(UdmSQLValue(&SQLres,i,0)));
						if(UDM_OK!=UdmDeleteURL(Indexer,&Doc,db))
							return(UDM_ERROR);
					}
					continue;
				}
				if(db->DBSQL_IN){
					urlin[0]=0;
					for(i=0;i<UdmSQLNumRows(&SQLres);i++){
						if(i)strcat(urlin,",");
						strcat(urlin,UdmSQLValue(&SQLres,i,0));
					}
					j+=i;
					switch(db->DBMode){
					case UDM_DBMODE_MULTI:
						for(i=MINDICT;i<MAXDICT;i++){
							if(last!=DICTNUM(i)){
								sprintf(qbuf,"DELETE FROM dict%d WHERE url_id in (%s)",DICTNUM(i),urlin);
								if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
									return rc;
								last=DICTNUM(i);
							}
						}
						break;
					case UDM_DBMODE_MULTI_CRC:
						for(i=MINDICT;i<MAXDICT;i++){
							if(last!=DICTNUM(i)){
								sprintf(qbuf,"DELETE FROM ndict%d WHERE url_id in (%s)",DICTNUM(i),urlin);
								if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
									return rc;
								last=DICTNUM(i);
							}
						}
						break;
					case UDM_DBMODE_CACHE:
						break;
					default:
						sprintf(qbuf,"DELETE FROM dict WHERE url_id in (%s)",urlin);
						if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
							return rc;
						break;
					}
					sprintf(qbuf,"DELETE FROM url WHERE rec_id in (%s)",urlin);
					if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
						return rc;
					
					sprintf(qbuf,"DELETE FROM urlinfo WHERE url_id in (%s)",urlin);
					if(UDM_OK!=(rc=UdmSQLQuery(db,NULL,qbuf)))
						return rc;
					UdmSQLFree(&SQLres);
				}else{
					for(i=0;i<UdmSQLNumRows(&SQLres);i++)
						UdmVarListReplaceInt(&Doc.Sections,"ID",atoi(UdmSQLValue(&SQLres,i,0)));
						if(UDM_OK!=UdmDeleteURL(Indexer,&Doc,db))
							return(UDM_ERROR);
					j+=i;
				}
			}else{
				UdmSQLFree(&SQLres);
				break;
			}
		}
	}
	return(UDM_OK);
}

/********************* Categories ************************************/
static int UdmCatList(UDM_AGENT * Indexer,UDM_CATEGORY *Cat,UDM_DB *db){
	size_t		i;
	char		qbuf[1024];
	UDM_SQLRES	SQLres;
	int		rc;
	
	if(db->DBType==UDM_DB_SAPDB){
		snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,lnk,name FROM categories WHERE path LIKE '%s__'",Cat->addr);
	}else{
		snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,link,name FROM categories WHERE path LIKE '%s__'",Cat->addr);
	}
	
	if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
		return rc;
	
	if(UdmSQLNumRows(&SQLres)){
		size_t nbytes,rows;
		
		nbytes=sizeof(UDM_CATITEM)*(UdmSQLNumRows(&SQLres)+Cat->ncategories);
		Cat->Category=(UDM_CATITEM*)realloc(Cat->Category,nbytes);
		rows=UdmSQLNumRows(&SQLres);
		for(i=0;i<rows;i++){
			UDM_CATITEM *r=&Cat->Category[Cat->ncategories+i];
			r[i].rec_id=atoi(UdmSQLValue(&SQLres,i,0));
			strcpy(r[i].path,UdmSQLValue(&SQLres,i,1));
			strcpy(r[i].link,UdmSQLValue(&SQLres,i,2));
			strcpy(r[i].name,UdmSQLValue(&SQLres,i,3));
		}
		Cat->ncategories+=rows;
	}
	UdmSQLFree(&SQLres);
	return UDM_OK;
}

static int UdmCatPath(UDM_AGENT *Indexer,UDM_CATEGORY *Cat,UDM_DB *db){
	size_t		i,l;
	char		qbuf[1024];
	int		rc;
	
	l=(strlen(Cat->addr)/2)+1;
	Cat->Category=(UDM_CATITEM*)realloc(Cat->Category,sizeof(UDM_CATITEM)*(l+Cat->ncategories));

	for(i=0;i<l;i++){
		char		head[128];
		UDM_SQLRES	SQLres;
		UDM_CATITEM	*r=&Cat->Category[Cat->ncategories];

		strncpy(head,Cat->addr,i*2);head[i*2]=0;

		if(db->DBType==UDM_DB_SAPDB){
			snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,lnk,name FROM categories WHERE path='%s'",head);
		}else{
			snprintf(qbuf,sizeof(qbuf)-1,"SELECT rec_id,path,link,name FROM categories WHERE path='%s'",head);
		}
		
		if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
			return rc;
		
		if(UdmSQLNumRows(&SQLres)){
			r[i].rec_id=atoi(UdmSQLValue(&SQLres,0,0));
			strcpy(r[i].path,UdmSQLValue(&SQLres,0,1));
			strcpy(r[i].link,UdmSQLValue(&SQLres,0,2));
			strcpy(r[i].name,UdmSQLValue(&SQLres,0,3));
		}
		UdmSQLFree(&SQLres);
	}
	Cat->ncategories+=l;
	return UDM_OK;
}


/******************* Search stuff ************************************/

static UDM_RESULT * UdmCloneList(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db){
	UDM_RESULT	*Res;
	size_t		i;
	char		qbuf[256];
	UDM_SQLRES	SQLres;
	int		crc32=UdmVarListFindInt(&Doc->Sections,"crc32",0);
	int		origin_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
	int		rc;
	
	sprintf(qbuf,"SELECT rec_id,url,content_type,last_mod_time,docsize FROM url WHERE crc32=%d AND (status=200 OR status=304 OR status=206) AND rec_id<>%d",crc32,origin_id);
	if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
		return(NULL);
	
	Res=UdmResultInit(NULL);
	Res->num_rows=UdmSQLNumRows(&SQLres);
	if(!(Res->num_rows))return(Res);
	if(Res->num_rows>5)Res->num_rows=5;
	
	Res->Doc=(UDM_DOCUMENT*)malloc(sizeof(UDM_DOCUMENT)*(Res->num_rows));
	
	for(i=0;i<Res->num_rows;i++){
		time_t		last_mod_time;
		char		buf[64];
		UDM_DOCUMENT	*D=&Res->Doc[i];
		
		UdmDocInit(D);
		UdmVarListAddInt(&D->Sections,"ID",atoi(UdmSQLValue(&SQLres,i,0)));
		UdmVarListAddStr(&D->Sections,"URL",UdmSQLValue(&SQLres,i,1));
		UdmVarListAddStr(&D->Sections,"Content-Type",UdmSQLValue(&SQLres,i,2));
		last_mod_time=atol(UdmSQLValue(&SQLres,i,3));
		UdmTime_t2HttpStr(last_mod_time, buf);
		UdmVarListAddStr(&D->Sections,"Last-Modified",buf);
		UdmVarListAddInt(&D->Sections,"Content-Length",atoi(UdmSQLValue(&SQLres,i,4)));
		UdmVarListAddInt(&D->Sections,"crc32",crc32);
		UdmVarListAddInt(&D->Sections,"Origin-ID",origin_id);
	}
	UdmSQLFree(&SQLres);
	return(Res);
}


/********************************************************/

static int UdmSortAndGroupByURL(UDM_AGENT *A,UDM_RESULT *R){
	unsigned long	ticks=UdmStartTimer();
	
	UdmLog(A,UDM_LOG_DEBUG,"Start sort by url_id %d words",R->CoordList.ncoords);
	UdmSortSearchWordsByURL(R->CoordList.Coords,R->CoordList.ncoords);
	ticks=UdmStartTimer()-ticks;
	UdmLog(A,UDM_LOG_DEBUG,"Stop sort by url_id:\t%.2f",(float)ticks/1000);
	
	UdmLog(A,UDM_LOG_DEBUG,"Start group by url_id %d docs",R->CoordList.ncoords);
	ticks=UdmStartTimer();
	UdmGroupByURL(A,R);
	ticks=UdmStartTimer()-ticks;
	UdmLog(A,UDM_LOG_DEBUG,"Stop group by url_id:\t%.2f",(float)ticks/1000);
	return UDM_OK;
}

static int UdmFindWordsBlob(UDM_AGENT *Agent, UDM_RESULT *Res,UDM_DB *db){
	int			wf[256];
	UdmWeightFactorsInit(UdmVarListFindStr(&Agent->Conf->Vars,"wf",""),wf);
	
#ifdef HAVE_MYSQL
if(db->DBDriver==UDM_DB_MYSQL){
	size_t			wordnum;
	UDM_URLCRDLISTLIST	Tmp;
	size_t			totalcoords=0;
	size_t			i,nbytes;
	UDM_SQLRES		*sqlres;
	
	nbytes=sizeof(UDM_SQLRES)*Res->WWList.nwords;
	sqlres=(UDM_SQLRES*)malloc(nbytes);
	bzero(sqlres,nbytes);
	bzero(&Tmp,sizeof(Tmp));
	
	for(wordnum=0;wordnum<Res->WWList.nwords;wordnum++){
		int		crc32,tablenum,numrows,rc;
		char		escwrd[128];
		char		qbuf[128];
		UDM_WIDEWORD	*W=&Res->WWList.Word[wordnum];
		MYSQL_ROW	row;
		unsigned long	ticks=UdmStartTimer();
		
		if (W->origin == UDM_WORD_ORIGIN_STOP) continue;
		crc32=UdmStrCRC32(W->word);
		tablenum=(unsigned)crc32 % db->numtables;
		UdmDBEscStr(db->DBType,escwrd,W->word,strlen(W->word));
		
		sprintf(qbuf,"SELECT tstamp,coords FROM bdict%02X WHERE word='%s'",tablenum,W->word);
		UdmLog(Agent,UDM_LOG_DEBUG,"%s",qbuf);
		if(UDM_OK!=(rc=UdmSQLQuery(db,&sqlres[wordnum],qbuf)))
			return rc;
		
		ticks=UdmStartTimer()-ticks;
		numrows=UdmSQLNumRows(&sqlres[wordnum]);
		UdmLog(Agent,UDM_LOG_DEBUG,"%d rows in result %.2f",numrows,(float)ticks/1000);
		
		ticks=UdmStartTimer();
		UdmLog(Agent,UDM_LOG_DEBUG,"Start fetching %d rows",numrows);
		
		while((row=mysql_fetch_row((sqlres[wordnum].mysqlres)))){
			unsigned long	*lengths=mysql_fetch_lengths(sqlres[wordnum].mysqlres);
			size_t		numcoords=lengths[1]/sizeof(UDM_URL_CRD);
			
			Tmp.List=(UDM_URLCRDLIST*)realloc(Tmp.List,(Tmp.nlists+1)*sizeof(Tmp.List[0]));
			Tmp.List[Tmp.nlists].Coords=(UDM_URL_CRD*)row[1];
			Tmp.List[Tmp.nlists].ncoords=numcoords;
			Tmp.List[Tmp.nlists].order=Res->WWList.Word[wordnum].order;
			Tmp.nlists++;
			totalcoords+=numcoords;
		}
		
		ticks=UdmStartTimer()-ticks;
		UdmLog(Agent,UDM_LOG_DEBUG,"Done %d coords %.2f",totalcoords,(float)ticks/1000);
	}
	
	Res->CoordList.Coords=(UDM_URL_CRD*)realloc(Res->CoordList.Coords,(Res->CoordList.ncoords+totalcoords)*(sizeof(UDM_URL_CRD)));
	
	for(i=0;i<Tmp.nlists;i++){
		UDM_URL_CRD	*Cb=Res->CoordList.Coords+Res->CoordList.ncoords;
		UDM_URL_CRD	*Ce=Cb+Tmp.List[i].ncoords;
		size_t		order=Tmp.List[i].order;
		
		memcpy(Cb,Tmp.List[i].Coords,Tmp.List[i].ncoords*sizeof(UDM_URL_CRD));
		
		for(;Cb<Ce;Cb++){
			uint4	coord=Cb->coord;
			uint4	section=UDM_WRDSEC(coord);
			uint4	weight=wf[section];
			
			coord=coord&0xFFFF0000;
			Cb->coord=coord+(weight<<8)+order;
		}
		Res->CoordList.ncoords+=Tmp.List[i].ncoords;
	}
	
	for(wordnum=0;wordnum<Res->WWList.nwords;wordnum++)
		UdmSQLFree(&sqlres[wordnum]);
	
	free(sqlres);
	free(Tmp.List);
	return UDM_OK;
}
#endif

#if defined HAVE_PGSQL || defined HAVE_IBASE
if(db->DBDriver==UDM_DB_PGSQL || db->DBDriver==UDM_DB_IBASE){
	size_t		wordnum,nbytes;
	UDM_SQLRES	*sqlres;
	size_t		totalcoords=0;
	int		rc;	
	
	nbytes=sizeof(UDM_SQLRES)*Res->WWList.nwords;
	sqlres=(UDM_SQLRES*)malloc(nbytes);
	bzero(sqlres,nbytes);
	
	for(wordnum=0;wordnum<Res->WWList.nwords;wordnum++){
		int		crc32,tablenum,numrows;
		char		escwrd[128];
		char		qbuf[128];
		UDM_WIDEWORD	*W=&Res->WWList.Word[wordnum];
		unsigned long	ticks=UdmStartTimer();
		size_t		row;
		
		if (W->origin == UDM_WORD_ORIGIN_STOP) continue;
		crc32=UdmStrCRC32(W->word);
		tablenum=(unsigned)crc32 % db->numtables;
		UdmDBEscStr(db->DBType,escwrd,W->word,strlen(W->word));
		
		sprintf(qbuf,"SELECT tstamp,ncoords,coords FROM bdict%02X WHERE word='%s'",tablenum,W->word);
		UdmLog(Agent,UDM_LOG_DEBUG,"%s",qbuf);
		if(UDM_OK!=(rc=UdmSQLQuery(db,&sqlres[wordnum],qbuf)))
			return rc;
		ticks=UdmStartTimer()-ticks;
		numrows=UdmSQLNumRows(&sqlres[wordnum]);
		UdmLog(Agent,UDM_LOG_DEBUG,"%d rows in result %.2f",numrows,(float)ticks/1000);
		
		ticks=UdmStartTimer();
		UdmLog(Agent,UDM_LOG_DEBUG,"Start fetching %d rows",numrows);
		
		for(row=0;row<numrows;row++){
			size_t	numcoords=atoi(UdmSQLValue(&sqlres[wordnum],row,1));
			totalcoords+=numcoords;
		}
		ticks=UdmStartTimer()-ticks;
		UdmLog(Agent,UDM_LOG_DEBUG,"Done %d coords %.2f",totalcoords,(float)ticks/1000);
	}
	
	Res->CoordList.Coords=(UDM_URL_CRD*)realloc(Res->CoordList.Coords,(Res->CoordList.ncoords+totalcoords)*(sizeof(UDM_URL_CRD)));
	
	for(wordnum=0;wordnum<Res->WWList.nwords;wordnum++){
		size_t	order=Res->WWList.Word[wordnum].order;
		size_t	row;
		int	numrows=UdmSQLNumRows(&sqlres[wordnum]);
		
		for(row=0;row<numrows;row++){
			size_t		numcoords=atoi(UdmSQLValue(&sqlres[wordnum],row,1));
			const char	*coords=UdmSQLValue(&sqlres[wordnum],row,2);
			size_t		ncoord=0;
			UDM_URL_CRD	*C=&Res->CoordList.Coords[Res->CoordList.ncoords];
			
			while(coords[0] && ncoord<numcoords){
				int	coord,section,weight;
				
				while(coords[0]=='{' || coords[0]==',')coords++;
				C[ncoord].url_id=atoi(coords);
				while(coords[0]!='{' && coords[0]!=',' && coords[0])coords++;
				
				while(coords[0]=='{' || coords[0]==',')coords++;
				coord=atoi(coords);
				while(coords[0]!='{' && coords[0]!=',' && coords[0])coords++;
				
				section=UDM_WRDSEC(coord);
				weight=wf[section];
				coord=coord&0xFFFF0000;
				C[ncoord].coord=coord+(weight<<8)+order;
				
				ncoord++;
			}
			
			/* 
			  Clear the tail if something is wrong: 
			  if bdict.ncoords doesn't correspond the
			  number of fetched word coordinates
			*/
			
			for(;ncoord<numcoords;ncoord++){
				C[ncoord].url_id=0;
				C[ncoord].coord=0;
			}
			
			Res->CoordList.ncoords+=numcoords;
		}
		UdmSQLFree(&sqlres[wordnum]);
	}
	free(sqlres);
	return UDM_OK;
}
#endif
	return UDM_OK;
}

static int UdmFindWordsSQL(UDM_AGENT * query,UDM_RESULT *Res,UDM_DB *db){
	char		qbuf[1024*4];
	size_t		wordnum;
	int		has_crosswrd=0;
	UDM_SQLRES	SQLres;
	int		word_match=UdmMatchMode(UdmVarListFindStr(&query->Conf->Vars,"wm","wrd"));
	int		wf[256];
	const char	*where=BuildWhere(query->Conf,db);
	int		use_crosswords=!strcasecmp(UdmVarListFindStr(&query->Conf->Vars,"CrossWords","no"),"yes");
	unsigned long 	ticks=UdmStartTimer();
	int		rc;
	
	UdmWeightFactorsInit(UdmVarListFindStr(&query->Conf->Vars,"wf",""),wf);
	
	/* Now find each word */
	for(wordnum=0;wordnum<Res->WWList.nwords;wordnum++){
		int	numrows,firstnum,curnum,tnum,tmin,tmax,tlst=-1;
		size_t	i;
		char	tablename[32]="dict";
		char	escwrd[1000];
		UDM_WIDEWORD	*W=&Res->WWList.Word[wordnum];
		
		if (W->origin == UDM_WORD_ORIGIN_STOP) continue;
		
		UdmDBEscStr(db->DBType,escwrd,W->word,strlen(W->word));

		if((db->DBMode==UDM_DBMODE_MULTI)&&(word_match!=UDM_MATCH_FULL)){
			/* This is for substring search!  */
			/* In Multi mode: we have to scan */
			/* almost all tables except those */
			/* with to short words            */
			
			tmin=DICTNUM(strlen(W->word));
			tmax=MAXDICT;
		}else{
			tmin=tmax=DICTNUM(strlen(W->word));
		}

		for(tnum=tmin;tnum<=tmax;tnum++){

			if(tlst!=DICTNUM(tnum)){
				tlst=DICTNUM(tnum);		

				ticks=UdmStartTimer();
				UdmLog(query,UDM_LOG_DEBUG,"Start search for '%s'",Res->WWList.Word[wordnum].word);
				
				switch(db->DBMode){
				case UDM_DBMODE_MULTI:
					sprintf(tablename,"dict%d",DICTNUM(tnum));
					break;
				case UDM_DBMODE_MULTI_CRC:
					sprintf(tablename,"ndict%d",DICTNUM(tnum));
					break;
				case UDM_DBMODE_SINGLE_CRC:
					strcpy(tablename,"ndict");
					break;
				default:
					break;
				}
				if((db->DBMode==UDM_DBMODE_SINGLE_CRC)||(db->DBMode==UDM_DBMODE_MULTI_CRC)){
					int crc;
					crc=W->crcword;
					if(where[0]){
						snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT %s.url_id,%s.intag \
FROM %s, url \
WHERE %s.word_id=%d \
AND url.rec_id=%s.url_id AND %s",
						tablename,tablename,
						tablename,tablename,
						crc,tablename,where);
					}else{
						snprintf(qbuf,sizeof(qbuf)-1,"SELECT url_id,intag FROM %s WHERE %s.word_id=%d",tablename,tablename,crc);
					}
				}else{
					char cmparg[256];
					switch(word_match){	
						case UDM_MATCH_BEGIN:
							sprintf(cmparg," LIKE '%s%%'",escwrd);
							break;
						case UDM_MATCH_END:
							sprintf(cmparg," LIKE '%%%s'",escwrd);
							break;
						case UDM_MATCH_SUBSTR:
							sprintf(cmparg," LIKE '%%%s%%'",escwrd);
							break;
						case UDM_MATCH_FULL:
						default:
							sprintf(cmparg,"='%s'",escwrd);
							break;
					}
					if(where[0]){
						snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT %s.url_id,%s.intag \
FROM %s, url \
WHERE %s.word%s \
AND url.rec_id=%s.url_id AND %s",
						tablename,tablename,
						tablename,tablename,
						cmparg,tablename,where);
					}else{
						snprintf(qbuf,sizeof(qbuf)-1,"SELECT url_id,intag FROM %s WHERE %s.word%s",tablename,tablename,cmparg);
					}
				}
				if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
					return rc;
					
				numrows=UdmSQLNumRows(&SQLres);
				ticks=UdmStartTimer()-ticks;
				UdmLog(query,UDM_LOG_DEBUG,"Stop search for '%s'\t%.2f  %d found",Res->WWList.Word[wordnum].word,(float)ticks/1000,numrows);
				
				/* Add new found word to the list */
				Res->CoordList.Coords=(UDM_URL_CRD*)UdmXrealloc(Res->CoordList.Coords,(Res->CoordList.ncoords+numrows)*sizeof(UDM_URL_CRD));
				
				firstnum=curnum=Res->CoordList.ncoords;
				for(i=0;i<numrows;i++){
					int url_id=0;
					uint4 coord=0;
					uint4 weight;
					uint4 section;
					int done=0;
#ifdef HAVE_MYSQL
					if(db->DBDriver==UDM_DB_MYSQL){
						MYSQL_ROW row;
						/* mysql_data_seek is slow */
						/* We will use sequential fetch instead*/
						row=mysql_fetch_row(SQLres.mysqlres);
						url_id=atoi(row[0]);
						coord=atoi(row[1]);
						done=1;
					}
#endif
					if(!done){
						url_id=atoi(UdmSQLValue(&SQLres,i,0));
						coord=atoi(UdmSQLValue(&SQLres,i,1));
					}

					section=UDM_WRDSEC(coord);
					weight=wf[section];
					if(weight){
						coord=coord&0xFFFF0000;
						Res->CoordList.Coords[curnum].url_id=url_id;
						Res->CoordList.Coords[curnum].coord=coord+(weight<<8)+Res->WWList.Word[wordnum].order;
						curnum++;
					}
				}
				UdmSQLFree(&SQLres);
				Res->WWList.Word[wordnum].count+=curnum-firstnum;
				Res->CoordList.ncoords=curnum;
				Res->CoordList.Coords=(UDM_URL_CRD*)UdmXrealloc(Res->CoordList.Coords,curnum*sizeof(UDM_URL_CRD));
			}
		}	
	}
	
	/* Now find each word in crosstable */
	has_crosswrd=((use_crosswords)&&(db->DBMode!=UDM_DBMODE_CACHE));
	for(wordnum=0;((has_crosswrd)&&(wordnum<Res->WWList.nwords));wordnum++){
		int		numrows,firstnum,curnum;
		char		tablename[32]="";
		char		escwrd[1000];
		size_t		i;
		UDM_WIDEWORD	*W=&Res->WWList.Word[wordnum];

		if(W->origin == UDM_WORD_ORIGIN_STOP) continue;

		UdmDBEscStr(db->DBType,escwrd,W->word,strlen(W->word));
		
		ticks=UdmStartTimer();
		UdmLog(query,UDM_LOG_DEBUG,"Start search for '%s'",W->word);
		
		switch(db->DBMode){
		case UDM_DBMODE_SINGLE_CRC:
		case UDM_DBMODE_MULTI_CRC:
			strcpy(tablename,"ncrossdict");
			break;
		case UDM_DBMODE_MULTI:
		default:
			strcpy(tablename,"crossdict");
			break;
		}
		if((db->DBMode==UDM_DBMODE_SINGLE_CRC)||
			(db->DBMode==UDM_DBMODE_MULTI_CRC)){
			int crc;
			crc=W->crcword;
			if(where[0]){
				snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT %s.url_id,%s.intag \
FROM %s, url \
WHERE %s.word_id=%d \
AND url.rec_id=%s.url_id AND %s",
				tablename,tablename,
				tablename,tablename,
				crc,tablename,where);
			}else{
				snprintf(qbuf,sizeof(qbuf)-1,"SELECT url_id,intag FROM %s WHERE %s.word_id=%d",tablename,tablename,crc);
			}
		}else{
			char cmparg[256];
			switch(word_match){	
				case UDM_MATCH_BEGIN:
					sprintf(cmparg," LIKE '%s%%'",escwrd);
					break;
				case UDM_MATCH_END:
					sprintf(cmparg," LIKE '%%%s'",escwrd);
					break;
				case UDM_MATCH_SUBSTR:
					sprintf(cmparg," LIKE '%%%s%%'",escwrd);
					break;
				case UDM_MATCH_FULL:
				default:
					sprintf(cmparg,"='%s'",escwrd);
					break;
			}
			if(where[0]){
		
				snprintf(qbuf,sizeof(qbuf)-1,"\
SELECT %s.url_id,%s.intag \
FROM %s, url \
WHERE %s.word%s \
AND url.rec_id=%s.url_id AND %s",
				tablename,tablename,
				tablename,tablename,
				cmparg,tablename,where);
			}else{
				snprintf(qbuf,sizeof(qbuf)-1,"SELECT url_id,intag FROM %s WHERE %s.word%s",tablename,tablename,cmparg);
			}
		}
		if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
			return rc;
		
		numrows=UdmSQLNumRows(&SQLres);
		ticks=UdmStartTimer()-ticks;
		UdmLog(query,UDM_LOG_DEBUG,"Stop search for '%s'\t%.2f %d found",Res->WWList.Word[wordnum].word,(float)ticks/1000,numrows);
		
		/* Add new found word to the list */
		Res->CoordList.Coords=(UDM_URL_CRD*)UdmXrealloc(Res->CoordList.Coords,(Res->CoordList.ncoords+numrows)*sizeof(UDM_URL_CRD));
		
		firstnum=curnum=Res->CoordList.ncoords;
		for(i=0;i<numrows;i++){
			int url_id=0;
			int coord=0;
			int done=0;
			int weight;
			int section;
#ifdef HAVE_MYSQL
			if(db->DBDriver==UDM_DB_MYSQL){
				MYSQL_ROW row;
				
				/* mysql_data_seek is slow */
				/* We will use sequential fetch instead*/
				
				row=mysql_fetch_row(SQLres.mysqlres);
				url_id=atoi(row[0]);
				coord=atoi(row[1]);
				done=1;
			}
#endif
			
			if(!done){
				url_id=atoi(UdmSQLValue(&SQLres,i,0));
				coord=atoi(UdmSQLValue(&SQLres,i,1));
			}
			
			section=UDM_WRDSEC(coord);
			weight=wf[section];
			if(weight){
				coord=coord&0xFFFF0000;
				Res->CoordList.Coords[curnum].url_id=url_id;
				Res->CoordList.Coords[curnum].coord=coord+(weight<<8)+Res->WWList.Word[wordnum].order;
				curnum++;
			}
		}
		UdmSQLFree(&SQLres);
		Res->CoordList.ncoords=curnum;
		Res->WWList.Word[wordnum].count+=curnum-firstnum;
		Res->CoordList.Coords=(UDM_URL_CRD*)UdmXrealloc(Res->CoordList.Coords,curnum*sizeof(UDM_URL_CRD));
	}
	return UDM_OK;
}


static int UdmTrack(UDM_AGENT * query,UDM_RESULT *Res,UDM_DB *db){
	char		qbuf[1024*4] = "";
	char		text_escaped[1024*4]="";
	const char	*words=UdmVarListFindStr(&query->Conf->Vars,"q-lc","");
	
	if (db->DBDriver == UDM_DB_FILES)
		return UDM_OK;
	
	/* Escape text to track it  */
	UdmDBEscStr(db->DBType, text_escaped, words, strlen(words));
	
	snprintf(qbuf,sizeof(qbuf)-1,"INSERT INTO qtrack (ip,qwords,qtime,found,ps,np,m,wm,o,t,cat,ul,wf,g,tmplt) VALUES ('%s','%s',%d,%d,%d,%d,'%s','%s',%d,'%s','%s','%s','%s','%s','%s')",
		UdmVarListFindStr(&query->Conf->Vars, "IP", ""),
		text_escaped,(int)time(NULL),Res->total_found,
		UdmVarListFindInt(&query->Conf->Vars, "ps", 20),
		UdmVarListFindInt(&query->Conf->Vars, "np", 0),
		UdmVarListFindStr(&query->Conf->Vars, "orig_m", ""),
		UdmVarListFindStr(&query->Conf->Vars, "wm", ""),
		UdmVarListFindInt(&query->Conf->Vars, "o", 0),
		UdmVarListFindStr(&query->Conf->Vars, "t", ""),
		UdmVarListFindStr(&query->Conf->Vars, "cat", ""),
		UdmVarListFindStr(&query->Conf->Vars, "ul", ""),
		UdmVarListFindStr(&query->Conf->Vars, "wf", ""),
		UdmVarListFindStr(&query->Conf->Vars, "g", ""),
		UdmVarListFindStr(&query->Conf->Vars, "tmplt", ""));
	
#ifdef DEBUG_SQL
	fprintf(stderr, "TrackQuery: %s\n", qbuf);
#endif
	return UdmSQLQuery(db,NULL,qbuf);
}

static void SQLResToDoc(UDM_DOCUMENT *D,UDM_SQLRES *sqlres,size_t i){
	time_t	last_mod_time;
	char	dbuf[64];
	
	UdmVarListAddStr(&D->Sections,"URL",UdmSQLValue(sqlres,i,1));
	UdmVarListAddStr(&D->Sections,"Content-Type",UdmSQLValue(sqlres,i,2));
	last_mod_time=atol(UdmSQLValue(sqlres,i,3));
	UdmTime_t2HttpStr(last_mod_time, dbuf);
	UdmVarListAddStr(&D->Sections,"Last-Modified",dbuf);
	UdmVarListAddStr(&D->Sections,"Content-Length",UdmSQLValue(sqlres,i,4));
	last_mod_time=atol(UdmSQLValue(sqlres,i,5));
	UdmTime_t2HttpStr(last_mod_time, dbuf);
	UdmVarListAddStr(&D->Sections,"Next-Index-Time",dbuf);
	UdmVarListAddInt(&D->Sections,"Referrer-ID",atoi(UdmSQLValue(sqlres,i,6)));
	UdmVarListAddInt(&D->Sections,"crc32",atoi(UdmSQLValue(sqlres,i,7)));
	UdmVarListAddStr(&D->Sections,"Tag",UdmSQLValue(sqlres,i,8));
	UdmVarListAddStr(&D->Sections,"Category",UdmSQLValue(sqlres,i,9));
	UdmVarListAddStr(&D->Sections,"Content-Language",UdmSQLValue(sqlres,i,10));
	UdmVarListAddStr(&D->Sections,"Charset",UdmSQLValue(sqlres,i,11));
}


static int UdmResAddDocInfo(UDM_AGENT *query, UDM_RESULT *Res,UDM_DB *db){
	size_t		i;
	char		instr[1024*4]="";
	char		qbuf[1024*4];
	UDM_SQLRES	SQLres;
	int		rc;
	
	if(!Res->num_rows)return UDM_OK;
	
	if(db->DBSQL_IN){
		size_t	j;
		
		/* Compose IN string and set to zero url_id field */
		for(i=0;i<Res->num_rows;i++){
			sprintf(UDM_STREND(instr),"%s%d",i?",":"",
				UdmVarListFindInt(&Res->Doc[i].Sections,"ID",0));
		}
		
		snprintf(qbuf,sizeof(qbuf),"SELECT rec_id,url,content_type,last_mod_time,docsize,next_index_time,referrer,crc32,tag,category,lang,charset FROM url WHERE rec_id IN (%s) ORDER BY rec_id",instr);
		if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
			return rc;
		
		for(j=0;j<Res->num_rows;j++) {
			for(i=0;i<UdmSQLNumRows(&SQLres);i++) {
				UDM_DOCUMENT	*D=&Res->Doc[j];
				int		url_id=UdmVarListFindInt(&D->Sections,"ID",0);
				
				if(url_id == atoi(UdmSQLValue(&SQLres,i,0))) {
					SQLResToDoc(D,&SQLres,i);
					break;
				}
			}
		}
		UdmSQLFree(&SQLres);
	}else{
		for(i=0;i<Res->num_rows;i++){
			UDM_DOCUMENT	*D=&Res->Doc[i];
			int		url_id=UdmVarListFindInt(&D->Sections,"ID",0);
			
			sprintf(qbuf,"SELECT rec_id,url,content_type,last_mod_time,docsize,next_index_time,referrer,crc32,tag,category,lang,charset FROM url WHERE rec_id=%d",url_id);
			if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
				return rc;
			
			if(UdmSQLNumRows(&SQLres)){
				SQLResToDoc(D,&SQLres,0);
			}
			UdmSQLFree(&SQLres);
		}
	}
	
	if(db->DBSQL_IN){
		size_t j;
		
		instr[0]='\0';
		/* Compose IN string and set to zero url_id field */
		for(i=0;i<Res->num_rows;i++){
			sprintf(UDM_STREND(instr),"%s%d",i?",":"",
				UdmVarListFindInt(&Res->Doc[i].Sections,"ID",0));
		}
		
		snprintf(qbuf,sizeof(qbuf),"SELECT url_id,sname,sval FROM urlinfo WHERE url_id IN (%s) ORDER BY url_id",instr);
		if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
			return rc;
		
		for(j=0;j<Res->num_rows;j++) {
			UDM_DOCUMENT	*D=&Res->Doc[j];
			int		url_id=UdmVarListFindInt(&D->Sections,"ID",0);
			
			for(i=0;i<UdmSQLNumRows(&SQLres);i++) {
				if(url_id == atoi(UdmSQLValue(&SQLres,i,0))){
					const char *sname=UdmSQLValue(&SQLres,i,1);
					const char *sval=UdmSQLValue(&SQLres,i,2);
					UdmVarListAddStr(&D->Sections,sname,sval);
				}
			}
		}
		UdmSQLFree(&SQLres);
	}else{
		for(i=0;i<Res->num_rows;i++){
			UDM_DOCUMENT	*D=&Res->Doc[i];
			size_t		row;
			int		url_id=UdmVarListFindInt(&D->Sections,"ID",0);
			
			sprintf(qbuf,"SELECT url_id,sname,sval FROM urlinfo WHERE url_id=%d",url_id);
			if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
				return rc;
			
			for(row=0;row<UdmSQLNumRows(&SQLres);row++) {
				const char *sname=UdmSQLValue(&SQLres,row,1);
				const char *sval=UdmSQLValue(&SQLres,row,2);
				UdmVarListAddStr(&D->Sections,sname,sval);
			}
			UdmSQLFree(&SQLres);
		}
	}
	return(UDM_OK);
}


static int UdmFindWords(UDM_AGENT * query,UDM_RESULT *Res,UDM_DB *db){
	const char	*cache_mode=NULL;
	int		page_size  =UdmVarListFindInt(&query->Conf->Vars,"ps",20);
	int		page_number=UdmVarListFindInt(&query->Conf->Vars,"np",0);
	size_t		topcount;
	unsigned long	ticks=UdmStartTimer();
	
	cache_mode  = UdmVarListFindStr(&query->Conf->Vars, "Cache", "no");
	UdmVarListAddStr(&query->Conf->Vars, "orig_m", UdmVarListFindStr(&query->Conf->Vars, "m", "all"));
	
	if( strcasecmp(cache_mode,"yes") || UdmSearchCacheFind(query,Res) ){
		/* If not found in search cache       */
		/* Let's do actual search now:        */
		/* Get groupped by url_id words array */
	
		if(db->DBMode==UDM_DBMODE_CACHE){
			UdmFindCache(query,Res);
		}else
		if(db->DBMode==UDM_DBMODE_BLOB){
			int res=UdmFindWordsBlob(query,Res,db);
			if(res!=UDM_OK)return res;
			UdmSortAndGroupByURL(query,Res);
		}else{
			int res=UdmFindWordsSQL(query,Res,db);
			if(res!=UDM_OK)return res;
			UdmSortAndGroupByURL(query,Res);
		}
		UdmLog(query,UDM_LOG_DEBUG,"Start sort by weight %d docs",Res->CoordList.ncoords);
		ticks=UdmStartTimer();
		
		/* Now sort results by the weight */
		
		topcount=page_size*(page_number+1)-1;
		if(topcount>=Res->CoordList.ncoords)topcount=Res->CoordList.ncoords-1;
		
		if(topcount<UDM_FAST_PRESORT_DOCS){
			UdmWrdTopSort(Res->CoordList.Coords,Res->CoordList.ncoords,topcount);
		}else{
			UdmSortSearchWordsByWeight(Res->CoordList.Coords,Res->CoordList.ncoords);
		}
		ticks=UdmStartTimer()-ticks;
		UdmLog(query,UDM_LOG_DEBUG,"Stop sort by weight:\t%.2f",(float)ticks/1000);
		
		/* Store into search result cache */
#if (WIN32|WINNT)
#else
		if((!strcasecmp(cache_mode,"yes"))&&(search_cache_size>-1)){
			fflush(stdout);
			fflush(stderr);
			if(fork()==0){
				/* Store to cache and exit program */
				UdmSearchCacheStore(query,Res);
				exit(0);
			}
		}
#endif
		Res->offset=1;
	}
	if(!strcasecmp(UdmVarListFindStr(&query->Conf->Vars, "TrackQuery", "no"),"yes")){
		UdmTrack(query,Res,db);
	}
	
	return UDM_OK;
}


int UdmFindSQL(UDM_AGENT *query,UDM_RESULT *Res){
	UDM_DB		*db=query->Conf->db;
	size_t		i;
	int		res=UDM_OK;
	int		page_size   = UdmVarListFindInt(&query->Conf->Vars, "ps", 20);
	int		page_number = UdmVarListFindInt(&query->Conf->Vars, "np", 0);
	unsigned long	ticks=UdmStartTimer();
	UDM_CHARSET	*lcs=UdmGetCharSet(UdmVarListFindStr(&query->Conf->Vars,"LocalCharset","iso-8859-1"));
	UDM_CHARSET	*bcs=UdmGetCharSet(UdmVarListFindStr(&query->Conf->Vars,"BrowserCharset","iso-8859-1"));
	
	if(!lcs)lcs=UdmGetCharSet("iso-8859-1");
	if(!bcs)bcs=UdmGetCharSet("iso-8859-1");
	
	UdmLog(query,UDM_LOG_DEBUG,"Start words stuff");	
	
	UdmPrepare(query,Res);				/* Prepare query    */
	AutoWild(query,&query->Conf->Vars);
	if(UDM_OK!=(res=UdmFindWords(query,Res,db)))	/* Load words array */
		goto ret;
	
	/* Return if no words found   */
	if(!Res->CoordList.ncoords)
		goto ret;
	
	/* Find curent page offsets */
	/* And If there was a request for */
	/* too large page number we will  */
	/* display the only one last doc  */
	
	Res->total_found=Res->CoordList.ncoords;
	Res->first=page_number*page_size;	
	if(Res->first>=Res->total_found)Res->first=Res->total_found-1;
	
	/* If results more than 1 page */
	/* we must cut the tail        */
	if((Res->first+page_size)>Res->total_found){
		Res->num_rows=Res->total_found-Res->first;
	}else{
		Res->num_rows=page_size;
	}
	Res->last=Res->first+Res->num_rows-1;
	
	UdmLog(query,UDM_LOG_DEBUG,"First: %d TotalFound: %d NumRows: %d",Res->first,Res->total_found,Res->num_rows);
	ticks=UdmStartTimer()-ticks;
	UdmLog(query,UDM_LOG_DEBUG,"Stop words stuff: %.2f",(float)ticks/1000);
	
	ticks=UdmStartTimer();
	UdmLog(query,UDM_LOG_DEBUG,"Start URLs");
	
	/* Allocate an array for documents information */
	Res->Doc=(UDM_DOCUMENT*)malloc(sizeof(UDM_DOCUMENT)*(Res->num_rows));
	
	/* Copy url_id and coord to result */
	for(i=0;i<Res->num_rows;i++){
		UdmDocInit(&Res->Doc[i]);
		UdmVarListAddInt(&Res->Doc[i].Sections,"ID",Res->CoordList.Coords[i+Res->first*Res->offset].url_id);
		UdmVarListAddInt(&Res->Doc[i].Sections,"Score", (int)Res->CoordList.Coords[i+Res->first*Res->offset].coord);
	}
	
	/* Load documents information */
	if(UDM_OK!=(res=UdmResAddDocInfo(query,Res,db)))
		goto ret;
	
	ticks=UdmStartTimer()-ticks;
	UdmLog(query,UDM_LOG_DEBUG,"Stop  URLs: %.2f",(float)ticks/1000);
	
	
	ticks=UdmStartTimer();
	UdmLog(query,UDM_LOG_DEBUG,"Start Clones");
	
	if(!strcasecmp(UdmVarListFindStr(&query->Conf->Vars,"DetectClones","yes"),"yes")){
		size_t num=Res->num_rows;
		for(i=0;i<num;i++){
			UDM_RESULT *Cl=UdmCloneList(query,&Res->Doc[i],db);
			if(Cl){
				Res->Doc=(UDM_DOCUMENT*)realloc(Res->Doc,sizeof(UDM_DOCUMENT)*(Res->num_rows+Cl->num_rows));
				memcpy(&Res->Doc[Res->num_rows],Cl->Doc,sizeof(UDM_DOCUMENT)*Cl->num_rows);
				Res->num_rows+=Cl->num_rows;
				free(Cl->Doc);
				free(Cl);
			}
		}
	}
	ticks=UdmStartTimer()-ticks;
	UdmLog(query,UDM_LOG_DEBUG,"Stop  Clones: %.2f",(float)ticks/1000);
	
	/* first and last begins from 0, make it begin from 1 */
	Res->first++;
	Res->last++;

	for(i=0;i<Res->num_rows;i++){
		UdmVarListReplaceInt(&Res->Doc[i].Sections,"Order",(int)(Res->first+i));
	}
	
ret:	
	strcpy(query->Conf->errstr,db->errstr);
	query->Conf->errcode=db->errcode;
	UdmResHlConvert(Res,lcs,bcs);
	return res;
}



/***********************************************************/
/*  HTDB stuff:  Indexing of database content              */
/***********************************************************/


static char * get_path_part(char *path,char *dst,int part){
	const char *s;
	char *e;
	int i=0;

	s=path;
	while(*s){
		if(i==part){
			if((e=strchr(s,'/')))strncpy(dst,s,(unsigned)(e-s));
			else	strcpy(dst,s);
			return(dst);
		}
		if(*s=='/')i++;
		s++;
	}
	*dst=0;
	return(dst);
}
static char * include_params(const char *src,char *path,char *dst){
	const char *s;
	char *e;
	int i;
	
	s=src;e=dst;*e=0;
	while(*s){
		if((*s=='\\')){
			*e=*(s+1);e++;*e=0;s+=2;
			continue;
		}
		if(*s!='$'){
			*e=*s;
			s++;e++;*e=0;
			continue;
		}
		s++;i=atoi(s);
		while((*s>='0')&&(*s<='9'))s++;
		get_path_part(path,e,i);
		while(*e)e++;
	}
	return(dst);
}


static int UdmHTDBGet(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,UDM_DB *db){
	char		qbuf[1024*4]="";
	char		*end=Doc->Buf.buf;
	UDM_SQLRES	SQLres;
	UDM_URL		realURL;
	const char	*url=UdmVarListFindStr(&Doc->Sections,"URL","");
	const char	*htdblist=UdmVarListFindStr(&Doc->Sections,"HTDBList","");
	const char	*htdbdoc=UdmVarListFindStr(&Doc->Sections,"HTDBDoc","");
	int		rc;
	
	Doc->Buf.buf[0]=0;
	UdmURLParse(&realURL,url);
	
	if(realURL.filename[0]){
		char real_path[1024]="";
		
		snprintf(real_path,sizeof(real_path)-1,"%s%s",realURL.path,realURL.filename);
		real_path[sizeof(real_path)-1]='\0';
		include_params(htdbdoc,real_path,qbuf);
		if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
			return rc;
		if(UdmSQLNumRows(&SQLres)==1){
			strcpy(Doc->Buf.buf,UdmSQLValue(&SQLres,0,0));
		}else{
			sprintf(Doc->Buf.buf,"HTTP/1.0 404 Not Found\r\n\r\n");
		}
		UdmSQLFree(&SQLres);
	}else{
		size_t	i;
		int	url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
		int	hops=UdmVarListFindInt(&Doc->Sections,"Hops",0);
		
		include_params(htdblist,realURL.path,qbuf);
		if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
			return rc;
		sprintf(Doc->Buf.buf,"HTTP/1.0 200 OK\r\nContent-type: text/html\r\n\r\n<HTML><BODY>\n");
		end=UDM_STREND(Doc->Buf.buf);
		
		for(i=0;i<UdmSQLNumRows(&SQLres);i++){
			UDM_HREF Href;
			Href.referrer=url_id;
			Href.stored=0;
			Href.tag=NULL;
			Href.hops=hops+1;
			Href.url=strdup(UdmSQLValue(&SQLres,i,0));
			Href.method=UDM_METHOD_GET;
			UdmHrefListAdd(&Doc->Hrefs,&Href);
			free(Href.url);
		}
		UdmSQLFree(&SQLres);
		strcpy(end,"</BODY></HTML>\n");
	}
	Doc->Buf.size=end-Doc->Buf.buf;
	return UDM_OK;
}

/************************** make index stuff *******************************/

UDM_UINT8_URLID* UdmLimit8SQL(UDM_AGENT *Agent,const char *field,int *count,int type,UDM_DB *db){
	char		qbuf[128];
	size_t		i,nrows;
	UDM_SQLRES	SQLres;
	UDM_UINT8_URLID	*data=NULL;
	int		rc;
	
	snprintf(qbuf,sizeof(qbuf)-1,"SELECT %s,rec_id FROM url",field);
	if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
		return NULL;
	nrows=UdmSQLNumRows(&SQLres);
	
	data=(UDM_UINT8_URLID*)malloc(nrows*sizeof(UDM_UINT8_URLID));
	if(!data){
		sprintf(db->errstr,"Error: %s",strerror(errno));
		db->errcode=1;
		return(NULL);
	}
	for(i=0;i<nrows;i++){
		const char *val0=UdmSQLValue(&SQLres,i,0);
		const char *val1=UdmSQLValue(&SQLres,i,1);

		switch(type){
			case 0: UdmDecodeHex8Str(val0,&data[i].hi,&data[i].lo); break;
			case 1: data[i].hi=atoi(val0); data[i].lo=0; break;
		}
		data[i].url_id=atoi(val1);
	}
	*count=nrows;
	UdmSQLFree(&SQLres);
	return(data);
}

UDM_UINT4_URLID* UdmLimit4SQL(UDM_AGENT *Agent,const char *field,int *count,int type,UDM_DB *db){
	char		qbuf[128];
	size_t		i,nrows;
	UDM_SQLRES	SQLres;
	UDM_UINT4_URLID	*data=NULL;
	int		rc;
	
	snprintf(qbuf,sizeof(qbuf)-1,"SELECT %s,rec_id FROM url",field);
	if(UDM_OK!=(rc=UdmSQLQuery(db,&SQLres,qbuf)))
		return NULL;
	
	nrows=UdmSQLNumRows(&SQLres);
	
	data=(UDM_UINT4_URLID*)malloc(nrows*sizeof(UDM_UINT4_URLID));
	if(!data){
		sprintf(db->errstr,"Error: %s",strerror(errno));
		db->errcode=0;
		return(NULL);
	}
	for(i=0;i<nrows;i++){
		const char *val0=UdmSQLValue(&SQLres,i,0);
		const char *val1=UdmSQLValue(&SQLres,i,1);

		switch(type){
			case 0: data[i].val=atoi(val0)/3600; break;
			case 1: data[i].val=atoi(val0)/60; break;
			case 2: {
				UDM_URL url;
				if(!UdmURLParse(&url,val0)){
					if(url.hostname) data[i].val=UdmStrCRC32(url.hostname);
					else data[i].val=0;
				}else
					data[i].val=0;
				break;
			case 3: data[i].val=UdmStrCRC32(val0);
			}
		}
		data[i].url_id=atoi(val1);
	}
	*count=nrows;
	UdmSQLFree(&SQLres);
	return(data);
}


/***************************************************************************/


int UdmURLActionSQL(UDM_AGENT * A, UDM_DOCUMENT * D, int cmd,UDM_DB *db){
	int res;

	switch(cmd){
		case UDM_URL_ACTION_DELETE:
			res=UdmDeleteURL(A,D,db);
			break;
			
		case UDM_URL_ACTION_ADD:
			res=UdmAddURL(A,D,db);
			break;
			
		case UDM_URL_ACTION_SUPDATE:
			res=UdmUpdateUrl(A,D,db);
			break;
			
		case UDM_URL_ACTION_LUPDATE:
			res=UdmLongUpdateURL(A,D,db);
			break;
			
		case UDM_URL_ACTION_INSWORDS:
			res=UdmStoreWords(A,D,db);
			break;
			
		case UDM_URL_ACTION_INSCWORDS:
			res=UdmStoreCrossWords(A,D,db);
			break;
			
		case UDM_URL_ACTION_DELWORDS:
			res=UdmDeleteWordFromURL(A,D,db);
			break;
			
		case UDM_URL_ACTION_DELCWORDS:
			res=UdmDeleteCrossWordFromURL(A,D,db);
			break;
			
		case UDM_URL_ACTION_UPDCLONE:
			res=UdmUpdateClone(A,D,db);
			break;
			
		case UDM_URL_ACTION_REGCHILD:
			res=UdmRegisterChild(A,D,db);
			break;
			
		case UDM_URL_ACTION_FINDBYURL:
			res=UdmFindURL(A,D,db);
			break;
			
		case UDM_URL_ACTION_FINDBYMSG:
			res=UdmFindMessage(A,D,db);
			break;
			
		case UDM_URL_ACTION_FINDORIG:
			res=UdmFindOrigin(A,D,db);
			break;
			
		case UDM_URL_ACTION_EXPIRE:
			res=UdmMarkForReindex(A,db);
			break;
			
		case UDM_URL_ACTION_REFERERS:
			res=UdmGetReferers(A,db);
			break;
		
		case UDM_URL_ACTION_DOCCOUNT:
			res=UdmGetDocCount(A,db);
			break;
		
		case UDM_URL_ACTION_HTDBGET:
			res=UdmHTDBGet(A,D,db);
			break;
		default:
			res=UDM_ERROR;
	}
	return res;
}

int UdmResActionSQL(UDM_AGENT *Agent, UDM_RESULT *Res, int cmd,UDM_DB *db){
	switch(cmd){
		case UDM_RES_ACTION_WORDS:
			return UdmFindWords(Agent,Res,db);
		case UDM_RES_ACTION_DOCINFO:
			return UdmResAddDocInfo(Agent,Res,db);
		case UDM_RES_ACTION_TARGETS:
			return UdmTargets(Agent,Res,db);
		case UDM_RES_ACTION_INSWORDS:
			return UdmResStoreBlob(Agent,Res,db);
		default:
			return UDM_ERROR;
	}
}

int UdmCatActionSQL(UDM_AGENT *Agent, UDM_CATEGORY *Cat, int cmd,UDM_DB *db){
	switch(cmd){
		case UDM_CAT_ACTION_LIST:
			return UdmCatList(Agent,Cat,db);
		case UDM_CAT_ACTION_PATH:
			return UdmCatPath(Agent,Cat,db);
		default:
			return UDM_ERROR;
	}
}

int UdmSrvActionSQL(UDM_AGENT *A, UDM_SERVERLIST *S, int cmd,UDM_DB *db){
	switch(cmd){
		case UDM_SRV_ACTION_TABLE:
			return UdmLoadServerTable(A,S,db);
		default:
			return UDM_ERROR;
	}
}

#endif /* HAVE_SQL */
